<?php
// stock_calculado_api.php - API independiente para llamadas AJAX
// Este archivo NO debe incluir templates ni HTML

// Prevenir cualquier output de HTML
ob_start();

// Configuración de errores
error_reporting(0);
ini_set('display_errors', 0);

// Headers JSON
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');

// Incluir configuración de base de datos
require_once '/home/crusardi/config_crusardi/database.php';

// Configurar charset UTF-8
if ($pdo) {
    $pdo->exec("SET NAMES 'utf8mb4'");
    $pdo->exec("SET CHARACTER SET utf8mb4");
    $pdo->exec("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");
}

// Limpiar cualquier output previo
ob_clean();

// Obtener acción
$action = $_GET['action'] ?? 'get_data';

// Respuesta por defecto
$response = [
    'success' => false,
    'message' => 'Acción no válida'
];

try {
    switch ($action) {
        case 'consolidar':
            // Ejecutar el procedimiento de consolidación
            if (!$pdo) {
                throw new Exception('No hay conexión a la base de datos');
            }
            
            // Ejecutar el procedimiento almacenado
            $stmt = $pdo->prepare("CALL consolidar_stock_crusardi()");
            $result = $stmt->execute();
            
            if (!$result) {
                throw new Exception('Error al ejecutar el procedimiento');
            }
            
            // Obtener el resumen si está disponible
            $resumen = [];
            try {
                do {
                    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    if ($rows && count($rows) > 0) {
                        $resumen = array_merge($resumen, $rows);
                    }
                } while ($stmt->nextRowset());
            } catch (Exception $e) {
                // Normal si no hay resultados
            }
            
            // Cerrar el statement
            $stmt->closeCursor();
            
            // Obtener estadísticas actualizadas
            $stats_sql = "SELECT 
                         COUNT(DISTINCT codigo) as total_productos,
                         SUM(stock_total) as total_unidades,
                         MAX(ultima_actualizacion) as ultima_actualizacion
                         FROM stock_calculado_cache 
                         WHERE stock_total > 0";
            $stats = $pdo->query($stats_sql)->fetch(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'message' => 'Stock consolidado exitosamente',
                'resumen' => $resumen,
                'stats' => $stats
            ];
            break;
            
        case 'get_data':
            // Obtener datos con filtros
            $search = $_GET['search'] ?? '';
            $tipo = $_GET['tipo'] ?? '';
            $categoria = $_GET['categoria'] ?? '';
            $tienda = $_GET['tienda'] ?? '';
            $page = max(1, intval($_GET['page'] ?? 1));
            $limit = 50;
            $offset = ($page - 1) * $limit;
            
            // Construir consulta
            $where = ['stock_total > 0'];
            $params = [];
            
            if ($search) {
                $where[] = "(search_text LIKE :search OR codigo LIKE :search2 OR nombre LIKE :search3)";
                $params[':search'] = "%$search%";
                $params[':search2'] = "%$search%";
                $params[':search3'] = "%$search%";
            }
            
            if ($tipo) {
                $where[] = "tipo = :tipo";
                $params[':tipo'] = $tipo;
            }
            
            if ($categoria) {
                $where[] = "(categoria1 = :cat OR categoria2 = :cat2)";
                $params[':cat'] = $categoria;
                $params[':cat2'] = $categoria;
            }
            
            if ($tienda) {
                $where[] = "tienda = :tienda";
                $params[':tienda'] = $tienda;
            }
            
            $where_clause = implode(' AND ', $where);
            
            // Obtener total
            $count_sql = "SELECT COUNT(DISTINCT codigo) FROM stock_calculado_cache WHERE $where_clause";
            $count_stmt = $pdo->prepare($count_sql);
            $count_stmt->execute($params);
            $total = $count_stmt->fetchColumn();
            
            // Obtener productos agrupados
            $sql = "SELECT 
                    codigo,
                    COALESCE(codigo_unico, codigo) as codigo_imagen,
                    MAX(nombre) as nombre,
                    tipo,
                    SUM(stock_total) as stock_total,
                    SUM(stock_sat) as stock_sat,
                    SUM(stock_outlet) as stock_outlet,
                    SUM(stock_showroom) as stock_showroom,
                    SUM(stock_own) as stock_own,
                    MAX(precio) as precio,
                    MAX(descuento) as descuento,
                    MIN(precio_final) as precio_final,
                    MAX(origen_descuento) as origen_descuento,
                    MAX(categoria1) as categoria1,
                    MAX(categoria2) as categoria2,
                    MAX(es_kit) as es_kit,
                    MAX(es_imperfecto) as es_imperfecto,
                    MAX(es_excepcion) as es_excepcion,
                    GROUP_CONCAT(DISTINCT CONCAT(COALESCE(bodega, 'CENTRAL'), ':', COALESCE(almacen, 'GENERAL')) SEPARATOR '|') as ubicaciones_detalle
                    FROM stock_calculado_cache 
                    WHERE $where_clause 
                    GROUP BY codigo, tipo
                    ORDER BY nombre ASC 
                    LIMIT $limit OFFSET $offset";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Procesar ubicaciones
            foreach ($productos as &$producto) {
                if ($producto['ubicaciones_detalle']) {
                    $ubicaciones = [];
                    $parts = explode('|', $producto['ubicaciones_detalle']);
                    foreach ($parts as $part) {
                        if (strpos($part, ':') !== false) {
                            list($bodega, $almacenes) = explode(':', $part);
                            $almacenes_array = array_filter(array_map('trim', explode(',', $almacenes)));
                            foreach ($almacenes_array as $almacen) {
                                $ubicaciones[] = $bodega . '-' . $almacen;
                            }
                        }
                    }
                    $producto['ubicaciones'] = implode(', ', array_unique($ubicaciones));
                } else {
                    $producto['ubicaciones'] = '';
                }
            }
            
            // Obtener estadísticas
            $stats_sql = "SELECT 
                          COUNT(DISTINCT codigo) as total_productos,
                          SUM(stock_total) as total_unidades,
                          SUM(CASE WHEN tipo = 'kit' THEN 1 ELSE 0 END) as total_kits,
                          SUM(CASE WHEN tipo = 'articulo' THEN 1 ELSE 0 END) as total_articulos,
                          SUM(CASE WHEN tipo = 'imperfecto' THEN 1 ELSE 0 END) as total_imperfectos,
                          SUM(CASE WHEN tipo = 'parte' THEN 1 ELSE 0 END) as total_partes,
                          COUNT(DISTINCT CASE WHEN descuento > 0 THEN codigo END) as con_descuento,
                          MAX(ultima_actualizacion) as ultima_actualizacion
                          FROM stock_calculado_cache 
                          WHERE stock_total > 0";
            $stats = $pdo->query($stats_sql)->fetch(PDO::FETCH_ASSOC);
            
            // Obtener categorías únicas
            $cat_sql = "SELECT DISTINCT categoria1 
                        FROM stock_calculado_cache 
                        WHERE categoria1 IS NOT NULL AND categoria1 != '' 
                        ORDER BY categoria1";
            $categorias = $pdo->query($cat_sql)->fetchAll(PDO::FETCH_COLUMN);
            
            // Obtener tiendas únicas
            $tiendas_sql = "SELECT DISTINCT tienda 
                            FROM stock_calculado_cache 
                            WHERE tienda IS NOT NULL AND tienda != '' 
                            ORDER BY tienda";
            $tiendas = $pdo->query($tiendas_sql)->fetchAll(PDO::FETCH_COLUMN);
            
            $response = [
                'success' => true,
                'productos' => $productos,
                'stats' => $stats,
                'categorias' => $categorias,
                'tiendas' => $tiendas,
                'pagination' => [
                    'total' => $total,
                    'page' => $page,
                    'pages' => ceil($total / $limit)
                ]
            ];
            break;
            
        case 'test':
            // Endpoint de prueba
            $response = [
                'success' => true,
                'message' => 'API funcionando correctamente',
                'timestamp' => date('Y-m-d H:i:s')
            ];
            break;
            
        default:
            $response = [
                'success' => false,
                'message' => 'Acción no reconocida: ' . $action
            ];
    }
    
} catch (PDOException $e) {
    $response = [
        'success' => false,
        'message' => 'Error de base de datos: ' . $e->getMessage(),
        'code' => $e->getCode()
    ];
} catch (Exception $e) {
    $response = [
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ];
}

// Asegurarse de que no hay output extra
ob_clean();

// Enviar respuesta JSON
echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
exit;
?>