<?php
// /stock/index.php

// SOLUCIÓN: Activar buffer de salida para evitar problemas con headers
ob_start();

require_once '/home/crusardi/config_crusardi/database.php';
require_once __DIR__ . '/vendor/autoload.php';

// Sesión
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verificar autenticación
if (!isset($_SESSION['user_email'])) {
    // Si es una petición AJAX, devolver JSON en lugar de redireccionar
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
        strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        
        header('Content-Type: application/json; charset=utf-8');
        header('HTTP/1.1 401 Unauthorized');
        echo json_encode([
            'success' => false, 
            'message' => 'Sesión expirada. Por favor, recargue la página.',
            'redirect' => 'auth/login.php'
        ]);
        exit;
    }
    
    // Si no es AJAX, redireccionar normalmente
    header('Location: auth/login.php');
    exit;
}

// Página actual
$page = $_GET['page'] ?? 'stock_calculado';
$allowed = [
    'stock_calculado',
    'importar_stock',
    'importar_combinaciones',
    'importar_productos',
    'cargar_imagenes',
    'descuentos',
    'excepciones',
    'gestionar_excepciones_kits',
    'productos_imperfectos',
    'upload_foto_imperfectos',
    'usuarios',
    'gestionar_categorias'
];
if (!in_array($page, $allowed, true)) {
    $page = 'stock_calculado';
}

// FUNCIONES DE SINCRONIZACIÓN DE CATEGORÍAS
function mapear_categoria_db($pdo, $grupo_articulo) {
    $grupo_upper = strtoupper(trim($grupo_articulo));
    
    // Buscar en la tabla de mapeo actualizada
    $stmt = $pdo->prepare("
        SELECT 
            c1.nombre as cat1, 
            c2.nombre as cat2
        FROM categoria_mapping cm
        LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
        LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
        WHERE cm.grupo_articulo = ? AND cm.activo = 1
    ");
    $stmt->execute([$grupo_upper]);
    $result = $stmt->fetch();
    
    if ($result) {
        return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
    }
    
    // Si no hay coincidencia exacta, buscar coincidencia parcial
    $stmt = $pdo->prepare("
        SELECT 
            c1.nombre as cat1, 
            c2.nombre as cat2
        FROM categoria_mapping cm
        LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
        LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
        WHERE ? LIKE CONCAT('%', cm.grupo_articulo, '%') AND cm.activo = 1 
        ORDER BY LENGTH(cm.grupo_articulo) DESC LIMIT 1
    ");
    $stmt->execute([$grupo_upper]);
    $result = $stmt->fetch();
    
    if ($result) {
        return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
    }
    
    // Default
    return ['cat1' => 'Otros', 'cat2' => 'General'];
}

// Gestión de acciones de categorías
if ($page === 'gestionar_categorias' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    // Tu código existente de gestión de categorías aquí
    // No lo modifico porque no tengo el código completo
}

// Gestión de acciones generales
if (isset($_REQUEST['action'])) {
    $action = basename($_REQUEST['action']);
    $action_file = __DIR__ . '/pages/actions/' . $action . '.php';
    if (file_exists($action_file)) {
        include $action_file;
        exit;
    }
}

// Inicializar variable de error de base de datos
$db_error = null;

// Incluir página solicitada
$page_path = __DIR__ . '/pages/' . $page . '.php';

// AHORA SÍ EMPIEZA EL HTML
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>Stock BoConcept - Crusardi</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; }
        
        /* Navegación dinámica mejorada */
        .sidebar-link {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }
        
        .sidebar-link::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            height: 100%;
            width: 0;
            background: linear-gradient(90deg, rgba(99, 102, 241, 0.1), rgba(99, 102, 241, 0.05));
            transition: width 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 0;
        }
        
        .sidebar-link:hover::before {
            width: 100%;
        }
        
        .sidebar-link:hover {
            transform: translateX(6px) scale(1.02);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }
        
        .sidebar-link.active {
            background: linear-gradient(135deg, #4f46e5, #6366f1);
            box-shadow: 0 8px 25px rgba(99, 102, 241, 0.3);
            transform: translateX(4px);
        }
        
        .sidebar-link.active::before {
            display: none;
        }
        
        /* Efectos para secciones colapsables */
        .nav-section {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .nav-section-header {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
        }
        
        .nav-section-header:hover {
            background: rgba(255, 255, 255, 0.05);
            transform: translateX(3px);
        }
        
        .nav-section-header.active {
            background: rgba(99, 102, 241, 0.1);
            border-left: 3px solid #6366f1;
        }
        
        .nav-section-content {
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .nav-section-content.show {
            animation: slideDown 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px) scale(0.95);
            }
            to {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }
        
        /* Indicadores de estado mejorados */
        .status-indicator {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: #10b981;
            box-shadow: 0 0 8px rgba(16, 185, 129, 0.5);
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; transform: translateY(-50%) scale(1); }
            50% { opacity: 0.7; transform: translateY(-50%) scale(1.2); }
        }
        
        /* Efectos de hover para iconos */
        .nav-icon {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .sidebar-link:hover .nav-icon {
            transform: scale(1.1) rotate(5deg);
            color: #6366f1;
        }
        
        .sidebar-link.active .nav-icon {
            transform: scale(1.1);
            color: #ffffff;
        }
        
        /* Animaciones para el sidebar */
        @keyframes slideIn {
            from { 
                transform: translateX(-100%); 
                opacity: 0; 
            }
            to { 
                transform: translateX(0); 
                opacity: 1; 
            }
        }
        
        .animate-slide-in {
            animation: slideIn 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        /* Mejoras para el contenido principal */
        .main-content {
            transition: margin-left 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        /* Efectos para botones de sección */
        .section-toggle {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
        }
        
        .section-toggle:hover {
            background: rgba(255, 255, 255, 0.08);
            transform: translateX(2px);
        }
        
        .section-toggle .chevron {
            transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .section-toggle.open .chevron {
            transform: rotate(180deg);
        }
        
        /* Asegurar que las tablas sean responsive */
        .table-container {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }
        
        /* Ajustes para el contenido principal */
        @media (min-width: 1024px) {
            .content-wrapper {
                max-width: 100%;
                width: 100%;
            }
        }
        
        /* Fix para contenido cortado */
        main > div {
            width: 100%;
            max-width: 100%;
        }
        
        /* Efectos de profundidad para el sidebar */
        .sidebar-depth {
            box-shadow: 
                0 10px 25px rgba(0, 0, 0, 0.2),
                0 4px 10px rgba(0, 0, 0, 0.1);
        }
        
        /* Gradientes dinámicos para elementos activos */
        .gradient-active {
            background: linear-gradient(135deg, #4f46e5, #6366f1, #8b5cf6);
            background-size: 200% 200%;
            animation: gradientShift 3s ease infinite;
        }
        
        @keyframes gradientShift {
            0%, 100% { background-position: 0% 50%; }
            50% { background-position: 100% 50%; }
        }
        
        /* Mejoras para el sidebar colapsable */
        .sidebar-collapsed {
            transform: translateX(-100%);
            opacity: 0;
            pointer-events: none;
        }
        
        .sidebar-expanded {
            transform: translateX(0);
            opacity: 1;
            pointer-events: auto;
        }
        
        /* Asegurar que el contenido principal se ajuste correctamente */
        .main-content-expanded {
            margin-left: 18rem; /* 72 * 0.25rem = 18rem */
        }
        
        .main-content-collapsed {
            margin-left: 0;
        }
        
        /* Transiciones más suaves para el colapso */
        .sidebar-transition {
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        /* Overlay para cuando el sidebar está colapsado en móvil */
        .sidebar-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 30;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .sidebar-overlay.active {
            opacity: 1;
            visibility: visible;
        }
    </style>
    <script src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js" defer></script>
</head>
<body class="bg-gray-50">

    <div x-data="{ 
        sidebarOpen: false,
        sidebarCollapsed: localStorage.getItem('sidebarCollapsed') === 'true'
    }" 
    x-init="$watch('sidebarCollapsed', val => localStorage.setItem('sidebarCollapsed', val))"
    class="relative min-h-screen flex">

        <!-- Overlay móvil -->
        <div x-show="sidebarOpen" 
             x-transition:enter="transition-opacity ease-linear duration-300"
             x-transition:enter-start="opacity-0"
             x-transition:enter-end="opacity-100"
             x-transition:leave="transition-opacity ease-linear duration-300"
             x-transition:leave-start="opacity-100"
             x-transition:leave-end="opacity-0"
             @click="sidebarOpen = false"
             class="fixed inset-0 bg-black bg-opacity-50 z-30 lg:hidden"></div>

        <!-- Sidebar -->
        <aside 
            class="fixed inset-y-0 left-0 bg-gradient-to-b from-gray-900 to-gray-800 text-white w-72 py-6 px-5 transform transition-all duration-300 ease-in-out z-40 shadow-xl sidebar-depth"
            :class="{
                'translate-x-0': sidebarOpen || !sidebarCollapsed,
                '-translate-x-full': !sidebarOpen && sidebarCollapsed
            }">
            
            <!-- Logo y título -->
            <div class="flex flex-col items-center mb-8 pb-6 border-b border-gray-700">
                <div class="bg-white rounded-xl p-3 mb-3 shadow-lg transform hover:scale-105 transition-transform duration-300">
                    <img src="assets/logo.png" alt="BoConcept" class="h-16 w-auto">
                </div>
                <h1 class="text-2xl font-bold bg-gradient-to-r from-indigo-400 to-purple-400 bg-clip-text text-transparent">Crusardi</h1>
                <span class="text-sm text-gray-400 mt-1">Sistema de Gestión de Stock</span>
            </div>

            <!-- Navegación -->
            <nav class="space-y-3">
                <?php if ($_SESSION['user_role'] === 'viewer'): ?>
                    <a href="index.php?page=stock_calculado" 
                       class="sidebar-link flex items-center px-4 py-3 rounded-lg relative <?php echo $page === 'stock_calculado' ? 'active gradient-active' : 'text-gray-300 hover:bg-gray-700 hover:text-white'; ?>">
                        <i class="fas fa-chart-line w-5 nav-icon"></i>
                        <span class="ml-3 font-medium relative z-10">Visor de Stock</span>
                        <?php if ($page === 'stock_calculado'): ?>
                            <div class="status-indicator"></div>
                        <?php endif; ?>
                    </a>
                <?php else: ?>
                    <!-- Vista principal -->
                    <a href="index.php?page=stock_calculado" 
                       class="sidebar-link flex items-center px-4 py-3 rounded-lg relative <?php echo $page === 'stock_calculado' ? 'active gradient-active' : 'text-gray-300 hover:bg-gray-700 hover:text-white'; ?>">
                        <i class="fas fa-chart-line w-5 nav-icon"></i>
                        <span class="ml-3 font-medium relative z-10">Visor de Stock</span>
                        <?php if ($page === 'stock_calculado'): ?>
                            <div class="status-indicator"></div>
                        <?php endif; ?>
                    </a>
                    
                    <!-- Sección de Importación -->
                    <div x-data="{ open: <?php echo in_array($page, ['importar_stock', 'importar_combinaciones', 'importar_productos']) ? 'true' : 'false'; ?> }" 
                         class="nav-section mt-6">
                        <button @click="open = !open" 
                                class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white transition-all duration-200"
                                :class="{ 'open': open, 'active': <?php echo in_array($page, ['importar_stock', 'importar_combinaciones', 'importar_productos']) ? 'true' : 'false'; ?> }">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-file-import w-5 nav-icon"></i>
                                <span class="font-medium">Importación</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div x-show="open" 
                             x-transition:enter="transition ease-out duration-300"
                             x-transition:enter-start="opacity-0 transform -translate-y-2"
                             x-transition:enter-end="opacity-100 transform translate-y-0"
                             class="nav-section-content mt-2 ml-8 space-y-1 show">
                            <a href="index.php?page=importar_stock" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'importar_stock' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-boxes w-4 mr-2 nav-icon"></i>Stock Físico
                                <?php if ($page === 'importar_stock'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                            <a href="index.php?page=importar_combinaciones" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'importar_combinaciones' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-box-open w-4 mr-2 nav-icon"></i>KITS / Recetas
                                <?php if ($page === 'importar_combinaciones'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                            <a href="index.php?page=importar_productos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'importar_productos' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-shopping-cart w-4 mr-2 nav-icon"></i>Productos
                                <?php if ($page === 'importar_productos'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                        </div>
                    </div>
                    
                    <!-- Sección de Configuración -->
                    <div x-data="{ open: <?php echo in_array($page, ['gestionar_categorias', 'cargar_imagenes', 'descuentos', 'excepciones', 'gestionar_excepciones_kits']) ? 'true' : 'false'; ?> }" 
                         class="nav-section mt-4">
                        <button @click="open = !open" 
                                class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white transition-all duration-200"
                                :class="{ 'open': open, 'active': <?php echo in_array($page, ['gestionar_categorias', 'cargar_imagenes', 'descuentos', 'excepciones', 'gestionar_excepciones_kits']) ? 'true' : 'false'; ?> }">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-cog w-5 nav-icon"></i>
                                <span class="font-medium">Configuración</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div x-show="open" 
                             x-transition:enter="transition ease-out duration-300"
                             x-transition:enter-start="opacity-0 transform -translate-y-2"
                             x-transition:enter-end="opacity-100 transform translate-y-0"
                             class="nav-section-content mt-2 ml-8 space-y-1 show">
                            <a href="index.php?page=gestionar_categorias" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'gestionar_categorias' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-tags w-4 mr-2 nav-icon"></i>Categorías
                                <?php if ($page === 'gestionar_categorias'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                            <a href="index.php?page=cargar_imagenes" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'cargar_imagenes' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-image w-4 mr-2 nav-icon"></i>Cargar Imágenes
                                <?php if ($page === 'cargar_imagenes'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                            <a href="index.php?page=descuentos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'descuentos' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-percent w-4 mr-2 nav-icon"></i>Descuentos
                                <?php if ($page === 'descuentos'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                            <a href="index.php?page=gestionar_excepciones_kits" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'importar_stock' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-exclamation-triangle w-4 mr-2 nav-icon"></i>Excepciones KITS
                                <?php if ($page === 'gestionar_excepciones_kits'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                        </div>
                    </div>
                    
                    <!-- Sección de Productos Especiales -->
                    <div x-data="{ open: <?php echo in_array($page, ['productos_imperfectos']) ? 'true' : 'false'; ?> }" 
                         class="nav-section mt-4">
                        <button @click="open = !open" 
                                class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white transition-all duration-200"
                                :class="{ 'open': open, 'active': <?php echo in_array($page, ['productos_imperfectos']) ? 'true' : 'false'; ?> }">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-box w-5 nav-icon"></i>
                                <span class="font-medium">Productos Especiales</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div x-show="open" 
                             x-transition:enter="transition ease-out duration-300"
                             x-transition:enter-start="opacity-0 transform -translate-y-2"
                             x-transition:enter-end="opacity-100 transform translate-y-0"
                             class="nav-section-content mt-2 ml-8 space-y-1 show">
                            <a href="index.php?page=productos_imperfectos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'productos_imperfectos' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-tools w-4 mr-2 nav-icon"></i>Productos Imperfectos
                                <?php if ($page === 'productos_imperfectos'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                        </div>
                    </div>
                    
                    <?php if ($_SESSION['user_role'] === 'admin'): ?>
                    <!-- Sección de Administración (solo admin) -->
                    <div x-data="{ open: <?php echo in_array($page, ['usuarios']) ? 'true' : 'false'; ?> }" 
                         class="nav-section mt-4">
                        <button @click="open = !open" 
                                class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white transition-all duration-200"
                                :class="{ 'open': open, 'active': <?php echo in_array($page, ['usuarios']) ? 'true' : 'false'; ?> }">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-shield-alt w-5 nav-icon"></i>
                                <span class="font-medium">Administración</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div x-show="open" 
                             x-transition:enter="transition ease-out duration-300"
                             x-transition:enter-start="opacity-0 transform -translate-y-2"
                             x-transition:enter-end="opacity-100 transform translate-y-0"
                             class="nav-section-content mt-2 ml-8 space-y-1 show">
                            <a href="index.php?page=usuarios" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'usuarios' ? 'active gradient-active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-users w-4 mr-2 nav-icon"></i>Usuarios
                                <?php if ($page === 'usuarios'): ?>
                                    <div class="status-indicator"></div>
                                <?php endif; ?>
                            </a>
                        </div>
                    </div>
                    <?php endif; ?>
                <?php endif; ?>
            </nav>

            <!-- Usuario -->
            <div class="absolute bottom-0 left-0 right-0 p-5 border-t border-gray-700">
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        <div class="bg-indigo-600 rounded-full p-2 mr-3 transform hover:scale-110 transition-transform duration-300">
                            <i class="fas fa-user text-white text-sm"></i>
                        </div>
                        <div>
                            <p class="text-sm font-medium text-white"><?php echo htmlspecialchars($_SESSION['user_name']); ?></p>
                            <p class="text-xs text-gray-400"><?php echo ucfirst($_SESSION['user_role']); ?></p>
                        </div>
                    </div>
                    <a href="auth/logout.php" class="text-red-400 hover:text-red-300 transition-colors transform hover:scale-110 duration-300">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </div>
        </aside>

        <!-- Contenido principal -->
        <div class="flex-1 flex flex-col transition-all duration-300"
             :class="{
                 'lg:ml-72': !sidebarCollapsed,
                 'lg:ml-0': sidebarCollapsed
             }">
            <!-- Header para desktop con botón de toggle -->
            <header class="bg-white shadow-sm p-4 flex justify-between items-center">
                <div class="flex items-center">
                    <!-- Botón para colapsar/expandir sidebar en desktop -->
                    <button @click="sidebarCollapsed = !sidebarCollapsed" 
                            class="hidden lg:block mr-4 text-gray-600 hover:text-gray-900 focus:outline-none">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                                  d="M4 6h16M4 12h16M4 18h16"></path>
                        </svg>
                    </button>
                    
                    <!-- Logo y título para móvil -->
                    <div class="flex items-center lg:hidden">
                        <img src="assets/logo.png" alt="BoConcept" class="h-10 w-auto mr-3">
                        <span class="text-lg font-semibold text-gray-800">Crusardi Stock</span>
                    </div>
                    
                    <!-- Título para desktop -->
                    <div class="hidden lg:block">
                        <h2 class="text-xl font-semibold text-gray-800">
                            <?php 
                            $page_titles = [
                                'stock_calculado' => 'Dashboard 🪑🟰➕',
                                'importar_stock' => 'Importar Stock Físico',
                                'importar_combinaciones' => 'Importar KITS / Recetas',
                                'importar_productos' => 'Importar Productos',
                                'cargar_imagenes' => 'Cargar Imágenes',
                                'descuentos' => 'Gestión de Descuentos',
                                'gestionar_excepciones_kits' => 'Excepciones de KITS',
                                'productos_imperfectos' => 'Productos Imperfectos',
                                'usuarios' => 'Gestión de Usuarios',
                                'gestionar_categorias' => 'Gestión de Categorías'
                            ];
                            echo $page_titles[$page] ?? 'Panel de Control';
                            ?>
                        </h2>
                    </div>
                </div>
                
                <!-- Botón menú móvil -->
                <button @click="sidebarOpen = !sidebarOpen" class="lg:hidden text-gray-600 hover:text-gray-900">
                    <i class="fas fa-bars text-xl"></i>
                </button>
            </header>

            <!-- Contenido de la página -->
            <main class="flex-1 overflow-y-auto">
                <div class="max-w-full mx-auto p-6">
                    <?php if ($db_error): ?>
                        <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-4">
                            <p class="font-bold">Error de conexión</p>
                            <p><?php echo htmlspecialchars($db_error); ?></p>
                        </div>
                    <?php elseif (file_exists($page_path)): ?>
                        <?php include $page_path; ?>
                    <?php else: ?>
                        <div class="bg-yellow-100 border-l-4 border-yellow-500 text-yellow-700 p-4">
                            <p class="font-bold">Página no encontrada</p>
                            <p>La página solicitada no existe.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <!-- Script personalizado para navegación dinámica -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Efectos de hover mejorados para enlaces
            const sidebarLinks = document.querySelectorAll('.sidebar-link');
            
            sidebarLinks.forEach(link => {
                link.addEventListener('mouseenter', function() {
                    this.style.transform = 'translateX(6px) scale(1.02)';
                });
                
                link.addEventListener('mouseleave', function() {
                    if (!this.classList.contains('active')) {
                        this.style.transform = 'translateX(0) scale(1)';
                    }
                });
            });
            
            // Efectos de ripple para botones de sección
            const sectionToggles = document.querySelectorAll('.section-toggle');
            
            sectionToggles.forEach(toggle => {
                toggle.addEventListener('click', function() {
                    // Agregar efecto de ripple
                    const ripple = document.createElement('div');
                    ripple.style.position = 'absolute';
                    ripple.style.borderRadius = '50%';
                    ripple.style.background = 'rgba(255, 255, 255, 0.3)';
                    ripple.style.transform = 'scale(0)';
                    ripple.style.animation = 'ripple 0.6s linear';
                    ripple.style.left = '50%';
                    ripple.style.top = '50%';
                    ripple.style.width = '20px';
                    ripple.style.height = '20px';
                    ripple.style.marginLeft = '-10px';
                    ripple.style.marginTop = '-10px';
                    
                    this.appendChild(ripple);
                    
                    setTimeout(() => {
                        ripple.remove();
                    }, 600);
                });
            });
            
            // Animación de entrada para elementos del sidebar
            const navItems = document.querySelectorAll('.nav-section, .sidebar-link');
            
            const observerOptions = {
                threshold: 0.1,
                rootMargin: '0px 0px -50px 0px'
            };
            
            const observer = new IntersectionObserver((entries) => {
                entries.forEach((entry, index) => {
                    if (entry.isIntersecting) {
                        setTimeout(() => {
                            entry.target.style.opacity = '1';
                            entry.target.style.transform = 'translateX(0)';
                        }, index * 100);
                    }
                });
            }, observerOptions);
            
            navItems.forEach((item, index) => {
                item.style.opacity = '0';
                item.style.transform = 'translateX(-20px)';
                item.style.transition = 'all 0.5s cubic-bezier(0.4, 0, 0.2, 1)';
                observer.observe(item);
            });
            
            // Efectos de profundidad en hover para el sidebar
            const sidebar = document.querySelector('aside');
            
            sidebar.addEventListener('mouseenter', function() {
                this.style.boxShadow = '0 20px 40px rgba(0, 0, 0, 0.3)';
            });
            
            sidebar.addEventListener('mouseleave', function() {
                this.style.boxShadow = '0 10px 25px rgba(0, 0, 0, 0.2), 0 4px 10px rgba(0, 0, 0, 0.1)';
            });
            
            // Indicadores de estado animados
            const statusIndicators = document.querySelectorAll('.status-indicator');
            
            statusIndicators.forEach(indicator => {
                indicator.addEventListener('mouseenter', function() {
                    this.style.animation = 'pulse 0.5s infinite';
                });
                
                indicator.addEventListener('mouseleave', function() {
                    this.style.animation = 'pulse 2s infinite';
                });
            });
            
            // Cerrar sidebar en móvil al hacer clic en un enlace
            const mobileSidebarLinks = document.querySelectorAll('.sidebar-link');
            mobileSidebarLinks.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth < 1024) {
                        // Buscar el elemento Alpine.js y cerrar el sidebar
                        const alpineElement = document.querySelector('[x-data]');
                        if (alpineElement && alpineElement.__x) {
                            alpineElement.__x.$data.sidebarOpen = false;
                        }
                    }
                });
            });
        });
        
        // Agregar estilos CSS dinámicos
        const style = document.createElement('style');
        style.textContent = `
            @keyframes ripple {
                to {
                    transform: scale(4);
                    opacity: 0;
                }
            }
            
            .sidebar-link:focus {
                outline: none;
                box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.3);
            }
            
            .section-toggle:focus {
                outline: none;
                box-shadow: 0 0 0 3px rgba(255, 255, 255, 0.2);
            }
            
            /* Mejoras para dispositivos táctiles */
            @media (hover: none) {
                .sidebar-link:hover {
                    transform: none;
                }
                
                .sidebar-link:active {
                    transform: translateX(3px) scale(1.01);
                }
            }
        `;
        document.head.appendChild(style);
    </script>

</body>
</html>
<?php
// SOLUCIÓN: Enviar el buffer al final
ob_end_flush();
?>