<?php
// /stock/index.php

// SOLUCIÓN: Activar buffer de salida para evitar problemas con headers
ob_start();

require_once '/home/crusardi/config_crusardi/database.php';
require_once __DIR__ . '/vendor/autoload.php';

// Sesión
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verificar autenticación
if (!isset($_SESSION['user_email'])) {
    // Si es una petición AJAX, devolver JSON en lugar de redireccionar
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
        strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        
        header('Content-Type: application/json; charset=utf-8');
        header('HTTP/1.1 401 Unauthorized');
        echo json_encode([
            'success' => false, 
            'message' => 'Sesión expirada. Por favor, recargue la página.',
            'redirect' => 'auth/login.php'
        ]);
        exit;
    }
    
    // Si no es AJAX, redireccionar normalmente
    header('Location: auth/login.php');
    exit;
}

// Página actual
$page = $_GET['page'] ?? 'stock_calculado';
$allowed = [
    'stock_calculado',
    'importar_stock',
    'importar_combinaciones',
    'importar_productos',
    'cargar_imagenes',
    'descuentos',
    'excepciones',
    'gestionar_excepciones_kits',
    'productos_imperfectos',
    'upload_foto_imperfectos',
    'usuarios',
    'gestionar_categorias'
];
if (!in_array($page, $allowed, true)) {
    $page = 'stock_calculado';
}

// FUNCIONES DE SINCRONIZACIÓN DE CATEGORÍAS
function mapear_categoria_db($pdo, $grupo_articulo) {
    $grupo_upper = strtoupper(trim($grupo_articulo));
    
    // Buscar en la tabla de mapeo actualizada
    $stmt = $pdo->prepare("
        SELECT 
            c1.nombre as cat1, 
            c2.nombre as cat2
        FROM categoria_mapping cm
        LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
        LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
        WHERE cm.grupo_articulo = ? AND cm.activo = 1
    ");
    $stmt->execute([$grupo_upper]);
    $result = $stmt->fetch();
    
    if ($result) {
        return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
    }
    
    // Si no hay coincidencia exacta, buscar coincidencia parcial
    $stmt = $pdo->prepare("
        SELECT 
            c1.nombre as cat1, 
            c2.nombre as cat2
        FROM categoria_mapping cm
        LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
        LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
        WHERE ? LIKE CONCAT('%', cm.grupo_articulo, '%') AND cm.activo = 1 
        ORDER BY LENGTH(cm.grupo_articulo) DESC LIMIT 1
    ");
    $stmt->execute([$grupo_upper]);
    $result = $stmt->fetch();
    
    if ($result) {
        return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
    }
    
    // Default
    return ['cat1' => 'Otros', 'cat2' => 'General'];
}

// Gestión de acciones de categorías
if ($page === 'gestionar_categorias' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    // Tu código existente de gestión de categorías aquí
    // No lo modifico porque no tengo el código completo
}

// Gestión de acciones generales
if (isset($_REQUEST['action'])) {
    $action = basename($_REQUEST['action']);
    $action_file = __DIR__ . '/pages/actions/' . $action . '.php';
    if (file_exists($action_file)) {
        include $action_file;
        exit;
    }
}

// Inicializar variable de error de base de datos
$db_error = null;

// Incluir página solicitada
$page_path = __DIR__ . '/pages/' . $page . '.php';

// AHORA SÍ EMPIEZA EL HTML
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Stock BoConcept - Crusardi</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { 
            font-family: 'Inter', sans-serif;
            -webkit-tap-highlight-color: transparent;
            -webkit-touch-callout: none;
        }
        
        /* Prevenir scroll cuando el sidebar está abierto en móvil */
        body.sidebar-open {
            overflow: hidden;
            position: fixed;
            width: 100%;
        }
        
        /* Navegación dinámica mejorada */
        .sidebar-link {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }
        
        .sidebar-link::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            height: 100%;
            width: 0;
            background: linear-gradient(90deg, rgba(99, 102, 241, 0.1), rgba(99, 102, 241, 0.05));
            transition: width 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 0;
        }
        
        @media (hover: hover) {
            .sidebar-link:hover::before {
                width: 100%;
            }
            
            .sidebar-link:hover {
                transform: translateX(6px) scale(1.02);
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            }
        }
        
        .sidebar-link.active {
            background: linear-gradient(135deg, #4f46e5, #6366f1);
            box-shadow: 0 8px 25px rgba(99, 102, 241, 0.3);
            transform: translateX(4px);
        }
        
        /* Efectos para secciones colapsables */
        .nav-section {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .nav-section-header {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
        }
        
        .nav-section-header:hover {
            background: rgba(255, 255, 255, 0.05);
            transform: translateX(3px);
        }
        
        .nav-section-header.active {
            background: rgba(99, 102, 241, 0.1);
            border-left: 3px solid #6366f1;
        }
        
        /* Indicadores de estado mejorados */
        .status-indicator {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: #10b981;
            box-shadow: 0 0 8px rgba(16, 185, 129, 0.5);
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; transform: translateY(-50%) scale(1); }
            50% { opacity: 0.7; transform: translateY(-50%) scale(1.2); }
        }
        
        /* Overlay para cuando el sidebar está colapsado en móvil */
        .sidebar-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 30;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .sidebar-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        /* Sidebar responsivo */
        .sidebar {
            position: fixed;
            top: 0;
            left: 0;
            bottom: 0;
            width: 18rem;
            transform: translateX(-100%);
            transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 40;
        }
        
        .sidebar.open {
            transform: translateX(0);
        }
        
        @media (min-width: 1024px) {
            .sidebar {
                transform: translateX(0);
            }
            
            .sidebar.collapsed {
                transform: translateX(-100%);
            }
            
            .sidebar-overlay {
                display: none;
            }
        }
        
        /* Asegurar que el sidebar esté visible en desktop por defecto */
        @media (min-width: 1024px) {
            .sidebar:not(.collapsed) {
                transform: translateX(0);
            }
        }
        
        /* Mejoras táctiles para iOS */
        @supports (-webkit-touch-callout: none) {
            .sidebar {
                -webkit-transform: translate3d(-100%, 0, 0);
                transform: translate3d(-100%, 0, 0);
                -webkit-backface-visibility: hidden;
                backface-visibility: hidden;
                -webkit-perspective: 1000px;
                perspective: 1000px;
            }
            
            .sidebar.open {
                -webkit-transform: translate3d(0, 0, 0);
                transform: translate3d(0, 0, 0);
            }
            
            /* Mejorar scroll en iOS */
            .scroll-container {
                -webkit-overflow-scrolling: touch;
                -webkit-transform: translateZ(0);
                transform: translateZ(0);
            }
        }
        
        /* Botón de cierre mejorado */
        .close-sidebar-btn {
            background: rgba(0, 0, 0, 0.3);
            border-radius: 50%;
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
            cursor: pointer;
            -webkit-tap-highlight-color: transparent;
        }
        
        .close-sidebar-btn:active {
            transform: scale(0.9);
            background: rgba(255, 255, 255, 0.3);
        }
        
        /* Asegurar que las tablas sean responsive */
        .table-container {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }
        
        /* Fix para scroll en iOS */
        .scroll-container {
            -webkit-overflow-scrolling: touch;
            overflow-y: auto;
        }
        
        /* Mejoras específicas para iPhone */
        @media screen and (max-width: 1023px) {
            /* Prevenir zoom en input en iOS */
            input, select, textarea {
                font-size: 16px !important;
            }
            
            /* Mejorar botones táctiles */
            button, .sidebar-link {
                min-height: 44px;
                min-width: 44px;
            }
            
            /* Mejorar scroll en sidebar */
            .sidebar nav {
                -webkit-overflow-scrolling: touch;
                scrollbar-width: none;
                -ms-overflow-style: none;
            }
            
            .sidebar nav::-webkit-scrollbar {
                display: none;
            }
        }
        
        /* Optimizaciones para Safari en iOS */
        @supports (-webkit-appearance: none) {
            .sidebar {
                -webkit-transform: translate3d(-100%, 0, 0);
                transform: translate3d(-100%, 0, 0);
            }
            
            .sidebar.open {
                -webkit-transform: translate3d(0, 0, 0);
                transform: translate3d(0, 0, 0);
            }
        }
    </style>
</head>
<body class="bg-gray-50">

    <div id="app" class="relative min-h-screen flex">

        <!-- Overlay móvil -->
        <div id="sidebar-overlay" class="sidebar-overlay"></div>

        <!-- Sidebar -->
        <aside id="sidebar" class="sidebar bg-gradient-to-b from-gray-900 to-gray-800 text-white py-6 px-5 shadow-xl">
            
            <!-- Logo y título -->
            <div class="flex flex-col items-center mb-8 pb-6 border-b border-gray-700 relative">
                <!-- Botón de cierre para móvil -->
                <button id="close-sidebar" class="lg:hidden absolute top-0 right-0 text-gray-400 hover:text-white p-2 close-sidebar-btn">
                    <i class="fas fa-times text-xl"></i>
                </button>
                
                <div class="bg-white rounded-xl p-3 mb-3 shadow-lg">
                    <img src="assets/logo.png" alt="BoConcept" class="h-16 w-auto">
                </div>
                <h1 class="text-2xl font-bold bg-gradient-to-r from-indigo-400 to-purple-400 bg-clip-text text-transparent">Crusardi</h1>
                <span class="text-sm text-gray-400 mt-1">Sistema de Gestión de Stock</span>
            </div>

            <!-- Navegación -->
            <nav class="space-y-3 scroll-container" style="max-height: calc(100vh - 300px); overflow-y: auto;">
                <?php if ($_SESSION['user_role'] === 'viewer'): ?>
                    <a href="index.php?page=stock_calculado" 
                       class="sidebar-link flex items-center px-4 py-3 rounded-lg relative <?php echo $page === 'stock_calculado' ? 'active' : 'text-gray-300 hover:bg-gray-700 hover:text-white'; ?>">
                        <i class="fas fa-chart-line w-5"></i>
                        <span class="ml-3 font-medium relative z-10">Visor de Stock</span>
                        <?php if ($page === 'stock_calculado'): ?>
                            <div class="status-indicator"></div>
                        <?php endif; ?>
                    </a>
                <?php else: ?>
                    <!-- Vista principal -->
                    <a href="index.php?page=stock_calculado" 
                       class="sidebar-link flex items-center px-4 py-3 rounded-lg relative <?php echo $page === 'stock_calculado' ? 'active' : 'text-gray-300 hover:bg-gray-700 hover:text-white'; ?>">
                        <i class="fas fa-chart-line w-5"></i>
                        <span class="ml-3 font-medium relative z-10">Visor de Stock</span>
                        <?php if ($page === 'stock_calculado'): ?>
                            <div class="status-indicator"></div>
                        <?php endif; ?>
                    </a>
                    
                    <!-- Sección de Importación -->
                    <div class="nav-section mt-6">
                        <button class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white"
                                data-section="import">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-file-import w-5"></i>
                                <span class="font-medium">Importación</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div class="nav-section-content mt-2 ml-8 space-y-1 hidden" data-content="import">
                            <a href="index.php?page=importar_stock" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'importar_stock' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-boxes w-4 mr-2"></i>Stock Físico
                            </a>
                            <a href="index.php?page=importar_combinaciones" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'importar_combinaciones' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-box-open w-4 mr-2"></i>KITS / Recetas
                            </a>
                            <a href="index.php?page=importar_productos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'importar_productos' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-shopping-cart w-4 mr-2"></i>Productos
                            </a>
                        </div>
                    </div>
                    
                    <!-- Sección de Configuración -->
                    <div class="nav-section mt-4">
                        <button class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white"
                                data-section="config">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-cog w-5"></i>
                                <span class="font-medium">Configuración</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div class="nav-section-content mt-2 ml-8 space-y-1 hidden" data-content="config">
                            <a href="index.php?page=gestionar_categorias" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'gestionar_categorias' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-tags w-4 mr-2"></i>Categorías
                            </a>
                            <a href="index.php?page=cargar_imagenes" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'cargar_imagenes' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-image w-4 mr-2"></i>Cargar Imágenes
                            </a>
                            <a href="index.php?page=descuentos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'descuentos' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-percent w-4 mr-2"></i>Descuentos
                            </a>
                            <a href="index.php?page=gestionar_excepciones_kits" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'gestionar_excepciones_kits' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-exclamation-triangle w-4 mr-2"></i>Excepciones KITS
                            </a>
                        </div>
                    </div>
                    
                    <!-- Sección de Productos Especiales -->
                    <div class="nav-section mt-4">
                        <button class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white"
                                data-section="special">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-box w-5"></i>
                                <span class="font-medium">Productos Especiales</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div class="nav-section-content mt-2 ml-8 space-y-1 hidden" data-content="special">
                            <a href="index.php?page=productos_imperfectos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'productos_imperfectos' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-tools w-4 mr-2"></i>Productos Imperfectos
                            </a>
                        </div>
                    </div>
                    
                    <?php if ($_SESSION['user_role'] === 'admin'): ?>
                    <!-- Sección de Administración (solo admin) -->
                    <div class="nav-section mt-4">
                        <button class="section-toggle w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white"
                                data-section="admin">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-shield-alt w-5"></i>
                                <span class="font-medium">Administración</span>
                            </div>
                            <i class="fas fa-chevron-down chevron transition-transform duration-300 text-sm"></i>
                        </button>
                        <div class="nav-section-content mt-2 ml-8 space-y-1 hidden" data-content="admin">
                            <a href="index.php?page=usuarios" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm relative <?php echo $page === 'usuarios' ? 'active' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-users w-4 mr-2"></i>Usuarios
                            </a>
                        </div>
                    </div>
                    <?php endif; ?>
                <?php endif; ?>
            </nav>

            <!-- Usuario -->
            <div class="absolute bottom-0 left-0 right-0 p-5 border-t border-gray-700">
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        <div class="bg-indigo-600 rounded-full p-2 mr-3">
                            <i class="fas fa-user text-white text-sm"></i>
                        </div>
                        <div>
                            <p class="text-sm font-medium text-white"><?php echo htmlspecialchars($_SESSION['user_name']); ?></p>
                            <p class="text-xs text-gray-400"><?php echo ucfirst($_SESSION['user_role']); ?></p>
                        </div>
                    </div>
                    <a href="auth/logout.php" class="text-red-400 hover:text-red-300 transition-colors">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </div>
        </aside>

        <!-- Contenido principal -->
        <div class="flex-1 flex flex-col transition-all duration-300 lg:ml-72" id="main-content">
            <!-- Header para desktop con botón de toggle -->
            <header class="bg-white shadow-sm p-4 flex justify-between items-center">
                <div class="flex items-center">
                    <!-- Botón para colapsar/expandir sidebar en desktop -->
                    <button id="toggle-sidebar-desktop" class="hidden lg:block mr-4 text-gray-600 hover:text-gray-900 focus:outline-none">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                                  d="M4 6h16M4 12h16M4 18h16"></path>
                        </svg>
                    </button>
                    
                    <!-- Logo y título para móvil -->
                    <div class="flex items-center lg:hidden">
                        <img src="assets/logo.png" alt="BoConcept" class="h-10 w-auto mr-3">
                        <span class="text-lg font-semibold text-gray-800">Crusardi Stock</span>
                    </div>
                    
                    <!-- Título para desktop -->
                    <div class="hidden lg:block">
                        <h2 class="text-xl font-semibold text-gray-800">
                            <?php 
                            $page_titles = [
                                'stock_calculado' => 'Calculadora de Stock',
                                'importar_stock' => 'Importar Stock Físico',
                                'importar_combinaciones' => 'Importar KITS / Recetas',
                                'importar_productos' => 'Importar Productos',
                                'cargar_imagenes' => 'Cargar Imágenes',
                                'descuentos' => 'Gestión de Descuentos',
                                'gestionar_excepciones_kits' => 'Excepciones de KITS',
                                'productos_imperfectos' => 'Productos Imperfectos',
                                'usuarios' => 'Gestión de Usuarios',
                                'gestionar_categorias' => 'Gestión de Categorías'
                            ];
                            echo $page_titles[$page] ?? 'Panel de Control';
                            ?>
                        </h2>
                    </div>
                </div>
                
                <!-- Botón menú móvil -->
                <button id="toggle-sidebar-mobile" class="lg:hidden text-gray-600 hover:text-gray-900">
                    <i class="fas fa-bars text-xl"></i>
                </button>
            </header>

            <!-- Contenido de la página -->
            <main class="flex-1 overflow-y-auto">
                <div class="max-w-full mx-auto p-6">
                    <?php if ($db_error): ?>
                        <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-4">
                            <p class="font-bold">Error de conexión</p>
                            <p><?php echo htmlspecialchars($db_error); ?></p>
                        </div>
                    <?php elseif (file_exists($page_path)): ?>
                        <?php include $page_path; ?>
                    <?php else: ?>
                        <div class="bg-yellow-100 border-l-4 border-yellow-500 text-yellow-700 p-4">
                            <p class="font-bold">Página no encontrada</p>
                            <p>La página solicitada no existe.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <!-- Script mejorado para manejo del sidebar -->
    <script>
                 (function() {
             // Variables del DOM
             const sidebar = document.getElementById('sidebar');
             const overlay = document.getElementById('sidebar-overlay');
             const mainContent = document.getElementById('main-content');
             const body = document.body;
             const toggleMobileBtn = document.getElementById('toggle-sidebar-mobile');
             const toggleDesktopBtn = document.getElementById('toggle-sidebar-desktop');
             const closeSidebarBtn = document.getElementById('close-sidebar');
             
             // Debug: verificar que los elementos existan
             console.log('Sidebar:', sidebar);
             console.log('Main content:', mainContent);
             console.log('Toggle mobile:', toggleMobileBtn);
             console.log('Toggle desktop:', toggleDesktopBtn);
            
            // Estado del sidebar
            let sidebarOpen = false;
            let sidebarCollapsed = localStorage.getItem('sidebarCollapsed') === 'true';
            
            // Funciones principales
            function openSidebar() {
                sidebarOpen = true;
                sidebar.classList.add('open');
                overlay.classList.add('active');
                body.classList.add('sidebar-open');
            }
            
            function closeSidebar() {
                sidebarOpen = false;
                sidebar.classList.remove('open');
                overlay.classList.remove('active');
                body.classList.remove('sidebar-open');
            }
            
            function toggleSidebarMobile() {
                if (sidebarOpen) {
                    closeSidebar();
                } else {
                    openSidebar();
                }
            }
            
            function toggleSidebarDesktop() {
                sidebarCollapsed = !sidebarCollapsed;
                localStorage.setItem('sidebarCollapsed', sidebarCollapsed);
                
                if (sidebarCollapsed) {
                    sidebar.classList.add('collapsed');
                    mainContent.classList.remove('lg:ml-72');
                    mainContent.classList.add('lg:ml-0');
                } else {
                    sidebar.classList.remove('collapsed');
                    mainContent.classList.add('lg:ml-72');
                    mainContent.classList.remove('lg:ml-0');
                }
            }
            
                         // Event listeners
             if (toggleMobileBtn) {
                 toggleMobileBtn.addEventListener('click', toggleSidebarMobile);
                 // Mejorar respuesta táctil en iOS
                 toggleMobileBtn.addEventListener('touchstart', function(e) {
                     e.preventDefault();
                     this.style.transform = 'scale(0.95)';
                 }, { passive: false });
                 
                 toggleMobileBtn.addEventListener('touchend', function(e) {
                     e.preventDefault();
                     this.style.transform = '';
                     toggleSidebarMobile();
                 }, { passive: false });
             }
            
            if (toggleDesktopBtn) {
                toggleDesktopBtn.addEventListener('click', toggleSidebarDesktop);
            }
            
            if (closeSidebarBtn) {
                closeSidebarBtn.addEventListener('click', closeSidebar);
            }
            
            if (overlay) {
                overlay.addEventListener('click', closeSidebar);
            }
            
            // Cerrar sidebar con tecla Escape
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && sidebarOpen) {
                    closeSidebar();
                }
            });
            
            // Cerrar sidebar al hacer clic en enlaces (móvil)
            const sidebarLinks = sidebar.querySelectorAll('a');
            sidebarLinks.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth < 1024) {
                        closeSidebar();
                    }
                });
            });
            
            // Manejo de secciones colapsables
            const sectionToggles = document.querySelectorAll('.section-toggle');
            sectionToggles.forEach(toggle => {
                toggle.addEventListener('click', function() {
                    const section = this.dataset.section;
                    const content = document.querySelector(`[data-content="${section}"]`);
                    const chevron = this.querySelector('.chevron');
                    
                    if (content) {
                        content.classList.toggle('hidden');
                        chevron.style.transform = content.classList.contains('hidden') ? '' : 'rotate(180deg)';
                    }
                });
            });
            
            // Inicializar secciones abiertas según la página actual
            <?php if (in_array($page, ['importar_stock', 'importar_combinaciones', 'importar_productos'])): ?>
            document.querySelector('[data-content="import"]')?.classList.remove('hidden');
            document.querySelector('[data-section="import"] .chevron').style.transform = 'rotate(180deg)';
            <?php endif; ?>
            
            <?php if (in_array($page, ['gestionar_categorias', 'cargar_imagenes', 'descuentos', 'gestionar_excepciones_kits'])): ?>
            document.querySelector('[data-content="config"]')?.classList.remove('hidden');
            document.querySelector('[data-section="config"] .chevron').style.transform = 'rotate(180deg)';
            <?php endif; ?>
            
            <?php if (in_array($page, ['productos_imperfectos'])): ?>
            document.querySelector('[data-content="special"]')?.classList.remove('hidden');
            document.querySelector('[data-section="special"] .chevron').style.transform = 'rotate(180deg)';
            <?php endif; ?>
            
            <?php if (in_array($page, ['usuarios'])): ?>
            document.querySelector('[data-content="admin"]')?.classList.remove('hidden');
            document.querySelector('[data-section="admin"] .chevron').style.transform = 'rotate(180deg)';
            <?php endif; ?>
            
                         // Función para inicializar el sidebar
             function initializeSidebar() {
                 console.log('Inicializando sidebar...');
                 console.log('Window width:', window.innerWidth);
                 console.log('Sidebar collapsed:', sidebarCollapsed);
                 
                 if (window.innerWidth >= 1024) {
                     if (!sidebarCollapsed) {
                         console.log('Desktop: sidebar visible');
                         mainContent.classList.add('lg:ml-72');
                         mainContent.classList.remove('lg:ml-0');
                         sidebar.classList.remove('collapsed');
                     } else {
                         console.log('Desktop: sidebar colapsado');
                         sidebar.classList.add('collapsed');
                         mainContent.classList.remove('lg:ml-72');
                         mainContent.classList.add('lg:ml-0');
                     }
                 } else {
                     console.log('Móvil: sidebar cerrado');
                     sidebar.classList.remove('open');
                     overlay.classList.remove('active');
                     body.classList.remove('sidebar-open');
                 }
             }
             
             // Inicializar sidebar cuando el DOM esté listo
             if (document.readyState === 'loading') {
                 document.addEventListener('DOMContentLoaded', initializeSidebar);
             } else {
                 initializeSidebar();
             }
            
                         // Manejar cambios de tamaño de ventana
             let resizeTimer;
             window.addEventListener('resize', function() {
                 clearTimeout(resizeTimer);
                 resizeTimer = setTimeout(function() {
                     console.log('Resize detectado, width:', window.innerWidth);
                     if (window.innerWidth >= 1024) {
                         closeSidebar();
                         // Re-inicializar para desktop
                         initializeSidebar();
                     } else {
                         // Asegurar que esté cerrado en móvil
                         closeSidebar();
                     }
                 }, 250);
             });
             
             // Manejar cambios de orientación en iPhone
             window.addEventListener('orientationchange', function() {
                 setTimeout(function() {
                     if (window.innerWidth < 1024) {
                         closeSidebar();
                     }
                     // Forzar reflow para evitar problemas de layout
                     sidebar.offsetHeight;
                 }, 100);
             });
            
                         // Prevenir scroll del body cuando el sidebar está abierto
             sidebar.addEventListener('touchmove', function(e) {
                 e.stopPropagation();
             }, { passive: true });
             
             // Soporte para gestos de swipe en móvil
             let touchStartX = 0;
             let touchEndX = 0;
             
             document.addEventListener('touchstart', function(e) {
                 touchStartX = e.changedTouches[0].screenX;
             }, { passive: true });
             
             document.addEventListener('touchend', function(e) {
                 touchEndX = e.changedTouches[0].screenX;
                 handleSwipe();
             }, { passive: true });
             
             function handleSwipe() {
                 const swipeThreshold = 50;
                 const swipeDistance = touchEndX - touchStartX;
                 
                 // Swipe derecho para abrir sidebar (solo en móvil)
                 if (swipeDistance > swipeThreshold && window.innerWidth < 1024 && !sidebarOpen) {
                     openSidebar();
                 }
                 // Swipe izquierdo para cerrar sidebar
                 else if (swipeDistance < -swipeThreshold && window.innerWidth < 1024 && sidebarOpen) {
                     closeSidebar();
                 }
             }
        })();
    </script>

</body>
</html>
<?php
// SOLUCIÓN: Enviar el buffer al final
ob_end_flush();
?>