<?php
// pages/upload_foto_imperfectos.php
// Manejador de upload de fotos para productos imperfectos

session_start();

// Verificar permisos
if (!isset($_SESSION['user_role']) || !in_array($_SESSION['user_role'], ['admin', 'editor'])) {
    $_SESSION['mensaje_error'] = "No tienes permisos para subir fotos";
    header('Location: ../index.php?page=productos_imperfectos');
    exit;
}

// Incluir configuración
require_once '/home/crusardi/config_crusardi/database.php';

// DEBUG temporal - eliminar después de verificar que funciona
$debug_log = "/tmp/upload_debug_" . date('Y-m-d') . ".log";
$debug_data = [
    'timestamp' => date('Y-m-d H:i:s'),
    'POST_data' => $_POST,
    'FILES_data' => isset($_FILES['foto']) ? [
        'name' => $_FILES['foto']['name'],
        'size' => $_FILES['foto']['size'],
        'error' => $_FILES['foto']['error']
    ] : 'NO_FILE'
];
file_put_contents($debug_log, print_r($debug_data, true) . "\n---\n", FILE_APPEND);

try {
    // Verificar y limpiar datos recibidos
    $codigo_unico = trim($_POST['codigo_unico'] ?? '');
    $foto_num = trim($_POST['foto_num'] ?? '');
    
    // VALIDACIÓN ESTRICTA
    if (empty($codigo_unico)) {
        throw new Exception("Código único no recibido o vacío");
    }
    
    if (empty($foto_num)) {
        throw new Exception("Número de foto no recibido o vacío");
    }
    
    // Validar que foto_num sea exactamente 1, 2, 3, 4 o 5
    if (!in_array($foto_num, ['1', '2', '3', '4', '5'])) {
        throw new Exception("Número de foto inválido: '$foto_num'. Debe ser 1, 2, 3, 4 o 5");
    }
    
    // Verificar archivo
    if (!isset($_FILES['foto'])) {
        throw new Exception("No se recibió el archivo 'foto'");
    }
    
    if ($_FILES['foto']['error'] !== UPLOAD_ERR_OK) {
        $errors = [
            UPLOAD_ERR_INI_SIZE => 'El archivo excede el tamaño máximo permitido por PHP',
            UPLOAD_ERR_FORM_SIZE => 'El archivo excede el tamaño máximo del formulario',
            UPLOAD_ERR_PARTIAL => 'El archivo se subió parcialmente',
            UPLOAD_ERR_NO_FILE => 'No se subió ningún archivo',
            UPLOAD_ERR_NO_TMP_DIR => 'Falta el directorio temporal',
            UPLOAD_ERR_CANT_WRITE => 'Error al escribir el archivo en disco',
            UPLOAD_ERR_EXTENSION => 'Extensión de archivo no permitida'
        ];
        $error_msg = $errors[$_FILES['foto']['error']] ?? 'Error desconocido: ' . $_FILES['foto']['error'];
        throw new Exception("Error en el archivo: $error_msg");
    }
    
    // Validar tipo de archivo
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png'];
    $file_type = $_FILES['foto']['type'];
    if (!in_array($file_type, $allowed_types)) {
        throw new Exception("Tipo de archivo no permitido: $file_type. Solo se permiten JPG, JPEG y PNG");
    }
    
    // Validar tamaño (máximo 5MB)
    if ($_FILES['foto']['size'] > 5 * 1024 * 1024) {
        throw new Exception("El archivo es muy grande. Máximo 5MB permitido");
    }
    
    file_put_contents($debug_log, "Validaciones OK - Código: $codigo_unico, Foto: $foto_num\n", FILE_APPEND);
    
    // CONEXIÓN FTP
    $conn = ftp_connect(FTP_HOST);
    if (!$conn) {
        throw new Exception("No se pudo conectar al servidor FTP");
    }
    
    if (!ftp_login($conn, FTP_USER, FTP_PASS)) {
        ftp_close($conn);
        throw new Exception("Error de autenticación FTP");
    }
    
    // Modo pasivo
    ftp_pasv($conn, true);
    
    // PREPARAR RUTAS
    $base_dir = '/boconcept/SALE2019';
    $product_dir = 'N ' . $codigo_unico;
    $full_product_path = $base_dir . '/' . $product_dir;
    $filename = $foto_num . '.jpg';  // AQUÍ ESTÁ LA CLAVE: usar el número exacto
    $full_file_path = $full_product_path . '/' . $filename;
    
    file_put_contents($debug_log, "Rutas preparadas:\n" .
                     "Base: $base_dir\n" .
                     "Directorio producto: $product_dir\n" .
                     "Archivo: $filename\n" .
                     "Ruta completa: $full_file_path\n", FILE_APPEND);
    
    // Ir al directorio base
    if (!ftp_chdir($conn, $base_dir)) {
        ftp_close($conn);
        throw new Exception("No se pudo acceder al directorio base: $base_dir");
    }
    
    // Crear directorio del producto si no existe
    $existing_dirs = ftp_nlist($conn, '.') ?: [];
    $dir_exists = false;
    
    foreach ($existing_dirs as $dir) {
        if (basename($dir) === $product_dir) {
            $dir_exists = true;
            break;
        }
    }
    
    if (!$dir_exists) {
        if (!ftp_mkdir($conn, $product_dir)) {
            ftp_close($conn);
            throw new Exception("No se pudo crear el directorio: $product_dir");
        }
        file_put_contents($debug_log, "Directorio creado: $product_dir\n", FILE_APPEND);
    } else {
        file_put_contents($debug_log, "Directorio ya existe: $product_dir\n", FILE_APPEND);
    }
    
    // SUBIR ARCHIVO CON NOMBRE ESPECÍFICO
    if (ftp_put($conn, $full_file_path, $_FILES['foto']['tmp_name'], FTP_BINARY)) {
        file_put_contents($debug_log, "Archivo subido exitosamente: $full_file_path\n", FILE_APPEND);
        
        // Verificar que el archivo se subió correctamente
        $uploaded_files = ftp_nlist($conn, $product_dir) ?: [];
        file_put_contents($debug_log, "Archivos en directorio después de subir: " . implode(', ', $uploaded_files) . "\n", FILE_APPEND);
        
        $_SESSION['mensaje_exito'] = "Foto $foto_num subida correctamente para el producto '$codigo_unico'";
        
    } else {
        ftp_close($conn);
        throw new Exception("Error al subir el archivo '$filename' al servidor FTP");
    }
    
    ftp_close($conn);
    
} catch (Exception $e) {
    $_SESSION['mensaje_error'] = "Error: " . $e->getMessage();
    file_put_contents($debug_log, "ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
    
    // Cerrar conexión FTP si está abierta
    if (isset($conn) && $conn) {
        ftp_close($conn);
    }
}

// Redirigir de vuelta
header('Location: ../index.php?page=productos_imperfectos');
exit;
?>