<?php
/**
 * Configuración del catálogo público BoConcept Crusardi
 */

// Configuración general
define('CATALOGO_PUBLICO_VERSION', '1.0.0');
define('CATALOGO_PUBLICO_UPDATE_INTERVAL', 6); // horas
define('CATALOGO_PUBLICO_MAX_PRODUCTS', 1000);

// URLs y enlaces
define('CATALOGO_PUBLICO_URL', 'https://stock.crusardi.net/catalogo_publico/');
define('IMAGENES_BASE_URL', 'https://imagenes.crusardi.net/boconcept/STOCKCOMERCIAL/');
define('IMAGEN_SIN_FOTO', 'https://imagenes.crusardi.net/boconcept/sin_foto.png');

// Contacto y WhatsApp
define('WHATSAPP_NUMBER', '573001234567');
define('EMAIL_CONTACTO', 'ventas@crusardi.net');
define('TELEFONO_CONTACTO', '+57 300 123 4567');

// Configuración de precios
define('PRECIO_CONSULTA_MIN', 999999);
define('DESCUENTO_MINIMO_DISPLAY', 5); // Porcentaje mínimo para mostrar descuento

// Configuración de filtros
$CATEGORIAS_DISPONIBLES = [
    'living' => 'Living',
    'dormitorio' => 'Dormitorio', 
    'oficina' => 'Oficina',
    'exterior' => 'Exterior',
    'cocina' => 'Cocina',
    'iluminacion' => 'Iluminación'
];

$RANGOS_PRECIO = [
    '0-500000' => 'Hasta $500.000',
    '500000-1000000' => '$500.000 - $1.000.000',
    '1000000-2000000' => '$1.000.000 - $2.000.000',
    '2000000-5000000' => '$2.000.000 - $5.000.000',
    '5000000+' => 'Más de $5.000.000',
    'consult' => 'Precio a consultar'
];

// Configuración de SEO
$SEO_CONFIG = [
    'title' => 'Catálogo de Productos BoConcept - Crusardi',
    'description' => 'Explora nuestra amplia gama de muebles BoConcept. Diseño danés contemporáneo, calidad excepcional. Envíos a toda Colombia.',
    'keywords' => 'BoConcept, muebles, diseño danés, sofás, sillas, mesas, decoración, Crusardi, Colombia',
    'og_title' => 'Catálogo BoConcept - Crusardi',
    'og_description' => 'Descubre la colección completa de muebles BoConcept. Diseño escandinavo, funcionalidad y estilo únicos.',
    'og_image' => 'https://stock.crusardi.net/assets/logo.png'
];

// Configuración de cache
$CACHE_CONFIG = [
    'enabled' => true,
    'duration' => 3600, // 1 hora en segundos
    'key_prefix' => 'catalogo_publico_'
];

// Configuración de logs
$LOG_CONFIG = [
    'enabled' => true,
    'directory' => __DIR__ . '/logs/',
    'max_files' => 30, // días
    'level' => 'INFO' // DEBUG, INFO, WARNING, ERROR
];

// Configuración de notificaciones
$NOTIFICATION_CONFIG = [
    'email_enabled' => true,
    'email_to' => 'tecnologia@crusardi.net',
    'email_from' => 'sistema@crusardi.net',
    'whatsapp_enabled' => false,
    'whatsapp_number' => WHATSAPP_NUMBER
];

// Función para obtener configuración
function getCatalogConfig($key = null) {
    global $CATEGORIAS_DISPONIBLES, $RANGOS_PRECIO, $SEO_CONFIG, $CACHE_CONFIG, $LOG_CONFIG, $NOTIFICATION_CONFIG;
    
    $config = [
        'categorias' => $CATEGORIAS_DISPONIBLES,
        'rangos_precio' => $RANGOS_PRECIO,
        'seo' => $SEO_CONFIG,
        'cache' => $CACHE_CONFIG,
        'logs' => $LOG_CONFIG,
        'notifications' => $NOTIFICATION_CONFIG
    ];
    
    if ($key === null) {
        return $config;
    }
    
    return isset($config[$key]) ? $config[$key] : null;
}

// Función para logging
function logCatalog($message, $level = 'INFO') {
    global $LOG_CONFIG;
    
    if (!$LOG_CONFIG['enabled']) {
        return;
    }
    
    $log_dir = $LOG_CONFIG['directory'];
    if (!is_dir($log_dir)) {
        mkdir($log_dir, 0755, true);
    }
    
    $log_file = $log_dir . 'catalogo_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] [$level] $message" . PHP_EOL;
    
    file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
}

// Función para formatear precio
function formatCatalogPrice($price) {
    if ($price <= 0 || $price > PRECIO_CONSULTA_MIN) {
        return 'Precio a consultar';
    }
    
    return '$' . number_format($price, 0, ',', '.');
}

// Función para obtener URL de imagen
function getCatalogImageUrl($codigo) {
    return IMAGENES_BASE_URL . $codigo . '.jpg';
}

// Función para verificar si es precio a consultar
function isCatalogPriceOnRequest($precio) {
    return $precio <= 0 || $precio > PRECIO_CONSULTA_MIN;
}

// Función para obtener descuento en porcentaje
function getCatalogDiscountPercent($precio, $precio_final) {
    if ($precio <= 0 || $precio_final <= 0) {
        return 0;
    }
    
    $descuento = (($precio - $precio_final) / $precio) * 100;
    return round($descuento);
}

// Función para determinar tipo de producto
function getCatalogProductType($producto) {
    if ($producto['es_imperfecto']) {
        return 'imperfecto';
    } elseif ($producto['es_kit']) {
        return 'kit';
    } elseif ($producto['tipo'] === 'articulo') {
        return 'articulo';
    }
    
    return 'otro';
}

// Función para obtener mensaje de WhatsApp
function getCatalogWhatsAppMessage($producto) {
    $mensaje = "Hola, me interesa el producto:\n\n";
    $mensaje .= "• " . $producto['nombre'] . "\n";
    $mensaje .= "• Referencia: " . $producto['codigo'] . "\n";
    
    if ($producto['codigo_unico']) {
        $mensaje .= "• Código único: " . $producto['codigo_unico'] . "\n";
    }
    
    $mensaje .= "• Stock disponible: " . number_format($producto['stock_total']) . " unidades\n\n";
    
    if (isCatalogPriceOnRequest($producto['precio'])) {
        $mensaje .= "Me gustaría consultar el precio y disponibilidad.\n\n";
    } else {
        $mensaje .= "Precio: " . formatCatalogPrice($producto['precio']) . "\n\n";
    }
    
    $mensaje .= "¿Podrían brindarme más información?";
    
    return urlencode($mensaje);
}

// Función para obtener estadísticas del catálogo
function getCatalogStats($pdo) {
    try {
        $query = "
            SELECT 
                COUNT(*) as total_productos,
                SUM(stock_total) as total_unidades,
                SUM(CASE WHEN tipo = 'kit' THEN 1 ELSE 0 END) as total_kits,
                SUM(CASE WHEN tipo = 'articulo' THEN 1 ELSE 0 END) as total_articulos,
                SUM(CASE WHEN es_imperfecto = 1 THEN 1 ELSE 0 END) as total_imperfectos,
                AVG(precio) as precio_promedio,
                MIN(precio) as precio_minimo,
                MAX(precio) as precio_maximo
            FROM stock_calculado_cache 
            WHERE stock_total > 0 AND precio > 0
        ";
        
        $stmt = $pdo->prepare($query);
        $stmt->execute();
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Formatear precios
        $stats['precio_promedio'] = round($stats['precio_promedio'], 0);
        $stats['precio_minimo'] = round($stats['precio_minimo'], 0);
        $stats['precio_maximo'] = round($stats['precio_maximo'], 0);
        
        return $stats;
        
    } catch (Exception $e) {
        logCatalog("Error obteniendo estadísticas: " . $e->getMessage(), 'ERROR');
        return [
            'total_productos' => 0,
            'total_unidades' => 0,
            'total_kits' => 0,
            'total_articulos' => 0,
            'total_imperfectos' => 0,
            'precio_promedio' => 0,
            'precio_minimo' => 0,
            'precio_maximo' => 0
        ];
    }
}
?>
