<?php
// pages/gestionar_excepciones_kits.php

// Verificar permisos
if (!isset($_SESSION['user_role']) || !in_array($_SESSION['user_role'], ['admin', 'editor'])) {
    echo '<div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg">
            <p class="text-red-700">No tienes permisos para acceder a esta sección.</p>
          </div>';
    return;
}

// Procesar acciones POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $response = ['success' => false, 'message' => ''];
    
    try {
        if (isset($_POST['action'])) {
            switch ($_POST['action']) {
                case 'add':
                    // Agregar nueva excepción
                    $stmt = $pdo->prepare("
                        INSERT INTO kit_component_exceptions 
                        (codigo_articulo, nombre_articulo, kit_referencia, kit_nombre, 
                         motivo_excepcion, etiqueta_como, precio_individual, created_by)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    
                    $stmt->execute([
                        $_POST['codigo_articulo'],
                        $_POST['nombre_articulo'] ?? null,
                        $_POST['kit_referencia'] ?? null,
                        $_POST['kit_nombre'] ?? null,
                        $_POST['motivo_excepcion'] ?? null,
                        $_POST['etiqueta_como'] ?? 'PARTE',
                        !empty($_POST['precio_individual']) ? $_POST['precio_individual'] : null,
                        $_SESSION['user_email']
                    ]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Excepción agregada correctamente';
                    break;
                    
                case 'update':
                    // Actualizar excepción
                    $stmt = $pdo->prepare("
                        UPDATE kit_component_exceptions 
                        SET codigo_articulo = ?, nombre_articulo = ?, kit_referencia = ?, 
                            kit_nombre = ?, motivo_excepcion = ?, etiqueta_como = ?, 
                            precio_individual = ?, activo = ?
                        WHERE id = ?
                    ");
                    
                    $stmt->execute([
                        $_POST['codigo_articulo'],
                        $_POST['nombre_articulo'] ?? null,
                        $_POST['kit_referencia'] ?? null,
                        $_POST['kit_nombre'] ?? null,
                        $_POST['motivo_excepcion'] ?? null,
                        $_POST['etiqueta_como'] ?? 'PARTE',
                        !empty($_POST['precio_individual']) ? $_POST['precio_individual'] : null,
                        isset($_POST['activo']) ? 1 : 0,
                        $_POST['id']
                    ]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Excepción actualizada correctamente';
                    break;
                    
                case 'delete':
                    // Eliminar excepción
                    $stmt = $pdo->prepare("DELETE FROM kit_component_exceptions WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Excepción eliminada correctamente';
                    break;
                    
                case 'toggle':
                    // Activar/desactivar excepción
                    $stmt = $pdo->prepare("
                        UPDATE kit_component_exceptions 
                        SET activo = NOT activo 
                        WHERE id = ?
                    ");
                    $stmt->execute([$_POST['id']]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Estado actualizado correctamente';
                    break;
            }
        }
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }
    
    // Si es petición AJAX, devolver JSON y terminar
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
        strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode($response);
        exit; // Terminar aquí para evitar incluir otras páginas
    }
}

// Obtener filtros
$filtro_busqueda = $_GET['busqueda'] ?? '';
$filtro_activo = $_GET['activo'] ?? 'todos';

// Construir consulta con filtros
$query = "SELECT * FROM kit_component_exceptions WHERE 1=1";
$params = [];

if (!empty($filtro_busqueda)) {
    $query .= " AND (codigo_articulo LIKE ? OR nombre_articulo LIKE ? OR kit_referencia LIKE ? OR kit_nombre LIKE ?)";
    $busqueda_param = "%$filtro_busqueda%";
    $params = array_merge($params, [$busqueda_param, $busqueda_param, $busqueda_param, $busqueda_param]);
}

if ($filtro_activo !== 'todos') {
    $query .= " AND activo = ?";
    $params[] = $filtro_activo === 'activos' ? 1 : 0;
}

$query .= " ORDER BY created_at DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$excepciones = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Obtener estadísticas
$stats = [
    'total' => count($excepciones),
    'activas' => 0,
    'inactivas' => 0
];

foreach ($excepciones as $exc) {
    if ($exc['activo']) {
        $stats['activas']++;
    } else {
        $stats['inactivas']++;
    }
}
?>

<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow-sm p-6">
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Gestión de Excepciones de Kits</h1>
                <p class="mt-1 text-sm text-gray-600">
                    Define componentes que deben tratarse de forma especial en los kits
                </p>
            </div>
            <button onclick="openModal('add')" 
                    class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                <i class="fas fa-plus mr-2"></i>Nueva Excepción
            </button>
        </div>
    </div>

    <!-- Estadísticas -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div class="bg-white rounded-lg shadow-sm p-6">
            <div class="flex items-center">
                <div class="p-3 bg-blue-100 rounded-lg">
                    <i class="fas fa-layer-group text-blue-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-600">Total Excepciones</p>
                    <p class="text-2xl font-bold text-gray-900"><?= number_format($stats['total']) ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-6">
            <div class="flex items-center">
                <div class="p-3 bg-green-100 rounded-lg">
                    <i class="fas fa-check-circle text-green-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-600">Activas</p>
                    <p class="text-2xl font-bold text-gray-900"><?= number_format($stats['activas']) ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-6">
            <div class="flex items-center">
                <div class="p-3 bg-gray-100 rounded-lg">
                    <i class="fas fa-times-circle text-gray-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-600">Inactivas</p>
                    <p class="text-2xl font-bold text-gray-900"><?= number_format($stats['inactivas']) ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Filtros -->
    <div class="bg-white rounded-lg shadow-sm p-6">
        <form method="GET" class="flex flex-wrap gap-4">
            <input type="hidden" name="page" value="gestionar_excepciones_kits">
            
            <div class="flex-1 min-w-[300px]">
                <label class="block text-sm font-medium text-gray-700 mb-2">Buscar</label>
                <input type="text" 
                       name="busqueda" 
                       value="<?= htmlspecialchars($filtro_busqueda) ?>"
                       placeholder="Código, nombre, referencia kit..."
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
            </div>
            
            <div class="w-48">
                <label class="block text-sm font-medium text-gray-700 mb-2">Estado</label>
                <select name="activo" 
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    <option value="todos" <?= $filtro_activo === 'todos' ? 'selected' : '' ?>>Todos</option>
                    <option value="activos" <?= $filtro_activo === 'activos' ? 'selected' : '' ?>>Activos</option>
                    <option value="inactivos" <?= $filtro_activo === 'inactivos' ? 'selected' : '' ?>>Inactivos</option>
                </select>
            </div>
            
            <div class="flex items-end gap-2">
                <button type="submit" 
                        class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                    <i class="fas fa-search mr-2"></i>Filtrar
                </button>
                <a href="?page=gestionar_excepciones_kits" 
                   class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
                    <i class="fas fa-redo mr-2"></i>Limpiar
                </a>
            </div>
        </form>
    </div>

    <!-- Tabla de excepciones -->
    <div class="bg-white rounded-lg shadow-sm overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Código Artículo
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Kit Referencia
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Etiqueta Como
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Precio Individual
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Estado
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Creado
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Acciones
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php if (empty($excepciones)): ?>
                        <tr>
                            <td colspan="7" class="px-6 py-4 text-center text-gray-500">
                                No se encontraron excepciones
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($excepciones as $excepcion): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div>
                                        <div class="text-sm font-medium text-gray-900">
                                            <?= htmlspecialchars($excepcion['codigo_articulo']) ?>
                                        </div>
                                        <?php if ($excepcion['nombre_articulo']): ?>
                                            <div class="text-sm text-gray-500">
                                                <?= htmlspecialchars($excepcion['nombre_articulo']) ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div>
                                        <div class="text-sm text-gray-900">
                                            <?= htmlspecialchars($excepcion['kit_referencia'] ?? '-') ?>
                                        </div>
                                        <?php if ($excepcion['kit_nombre']): ?>
                                            <div class="text-sm text-gray-500">
                                                <?= htmlspecialchars($excepcion['kit_nombre']) ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full 
                                        <?= $excepcion['etiqueta_como'] === 'KIT' ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-800' ?>">
                                        <?= htmlspecialchars($excepcion['etiqueta_como']) ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    <?= $excepcion['precio_individual'] ? '$' . number_format($excepcion['precio_individual'], 0) : '-' ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <button onclick="toggleStatus(<?= $excepcion['id'] ?>)" 
                                            class="relative inline-flex items-center h-6 rounded-full w-11 transition-colors
                                            <?= $excepcion['activo'] ? 'bg-green-500' : 'bg-gray-200' ?>">
                                        <span class="inline-block w-4 h-4 transform transition-transform bg-white rounded-full shadow
                                            <?= $excepcion['activo'] ? 'translate-x-6' : 'translate-x-1' ?>">
                                        </span>
                                    </button>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    <div><?= date('d/m/Y', strtotime($excepcion['created_at'])) ?></div>
                                    <div class="text-xs"><?= htmlspecialchars($excepcion['created_by'] ?? '') ?></div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                    <button onclick='openModal("edit", <?= json_encode($excepcion) ?>)' 
                                            class="text-indigo-600 hover:text-indigo-900 mr-3">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button onclick="deleteException(<?= $excepcion['id'] ?>)" 
                                            class="text-red-600 hover:text-red-900">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Información adicional -->
    <div class="bg-blue-50 border-l-4 border-blue-400 p-4">
        <div class="flex">
            <div class="flex-shrink-0">
                <i class="fas fa-info-circle text-blue-400"></i>
            </div>
            <div class="ml-3">
                <h3 class="text-sm font-medium text-blue-800">Información sobre excepciones</h3>
                <div class="mt-2 text-sm text-blue-700">
                    <ul class="list-disc list-inside space-y-1">
                        <li><strong>PARTE</strong>: El componente se muestra como parte individual del kit</li>
                        <li><strong>KIT</strong>: El componente se etiqueta como kit completo</li>
                        <li><strong>Precio Individual</strong>: Si se especifica, se usa este precio en lugar del precio del sistema</li>
                        <li>Las excepciones solo aplican cuando están activas</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal para agregar/editar -->
<div id="exceptionModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <form id="exceptionForm" class="p-6">
                <input type="hidden" name="action" id="modalAction">
                <input type="hidden" name="id" id="exceptionId">
                
                <h2 class="text-xl font-bold text-gray-900 mb-4" id="modalTitle">Nueva Excepción</h2>
                
                <div class="space-y-4">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Código Artículo <span class="text-red-500">*</span>
                            </label>
                            <input type="text" 
                                   name="codigo_articulo" 
                                   id="codigo_articulo"
                                   required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Nombre Artículo
                            </label>
                            <input type="text" 
                                   name="nombre_articulo" 
                                   id="nombre_articulo"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Kit Referencia
                            </label>
                            <input type="text" 
                                   name="kit_referencia" 
                                   id="kit_referencia"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Kit Nombre
                            </label>
                            <input type="text" 
                                   name="kit_nombre" 
                                   id="kit_nombre"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Motivo de Excepción
                        </label>
                        <textarea name="motivo_excepcion" 
                                  id="motivo_excepcion"
                                  rows="3"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500"></textarea>
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Etiquetar Como
                            </label>
                            <select name="etiqueta_como" 
                                    id="etiqueta_como"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                                <option value="PARTE">PARTE</option>
                                <option value="KIT">KIT</option>
                            </select>
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Precio Individual (opcional)
                            </label>
                            <input type="number" 
                                   name="precio_individual" 
                                   id="precio_individual"
                                   step="0.01"
                                   min="0"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                    </div>
                    
                    <div class="flex items-center" id="activoContainer" style="display: none;">
                        <input type="checkbox" 
                               name="activo" 
                               id="activo"
                               class="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300 rounded">
                        <label for="activo" class="ml-2 block text-sm text-gray-700">
                            Excepción activa
                        </label>
                    </div>
                </div>
                
                <div class="mt-6 flex justify-end space-x-3">
                    <button type="button" 
                            onclick="closeModal()"
                            class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
                        Cancelar
                    </button>
                    <button type="submit" 
                            class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                        <i class="fas fa-save mr-2"></i>Guardar
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Funciones del modal
function openModal(action, data = null) {
    const modal = document.getElementById('exceptionModal');
    const form = document.getElementById('exceptionForm');
    const title = document.getElementById('modalTitle');
    const actionInput = document.getElementById('modalAction');
    const activoContainer = document.getElementById('activoContainer');
    
    // Resetear formulario
    form.reset();
    
    // Configurar según la acción
    if (action === 'add') {
        title.textContent = 'Nueva Excepción';
        actionInput.value = 'add';
        activoContainer.style.display = 'none';
    } else if (action === 'edit' && data) {
        title.textContent = 'Editar Excepción';
        actionInput.value = 'update';
        activoContainer.style.display = 'flex';
        
        // Llenar campos
        document.getElementById('exceptionId').value = data.id;
        document.getElementById('codigo_articulo').value = data.codigo_articulo || '';
        document.getElementById('nombre_articulo').value = data.nombre_articulo || '';
        document.getElementById('kit_referencia').value = data.kit_referencia || '';
        document.getElementById('kit_nombre').value = data.kit_nombre || '';
        document.getElementById('motivo_excepcion').value = data.motivo_excepcion || '';
        document.getElementById('etiqueta_como').value = data.etiqueta_como || 'PARTE';
        document.getElementById('precio_individual').value = data.precio_individual || '';
        document.getElementById('activo').checked = data.activo == 1;
    }
    
    modal.classList.remove('hidden');
}

function closeModal() {
    document.getElementById('exceptionModal').classList.add('hidden');
}

// Manejar envío del formulario
document.getElementById('exceptionForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    fetch('?page=gestionar_excepciones_kits', {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Error al procesar la solicitud');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error al procesar la solicitud');
    });
});

// Función para cambiar estado
function toggleStatus(id) {
    if (!confirm('¿Desea cambiar el estado de esta excepción?')) return;
    
    const formData = new FormData();
    formData.append('action', 'toggle');
    formData.append('id', id);
    
    fetch('?page=gestionar_excepciones_kits', {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Error al cambiar el estado');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error al cambiar el estado');
    });
}

// Función para eliminar
function deleteException(id) {
    if (!confirm('¿Está seguro de eliminar esta excepción?')) return;
    
    const formData = new FormData();
    formData.append('action', 'delete');
    formData.append('id', id);
    
    fetch('?page=gestionar_excepciones_kits', {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Error al eliminar');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error al eliminar');
    });
}

// Cerrar modal al hacer clic fuera
document.getElementById('exceptionModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeModal();
    }
});
</script>