<?php
/**
 * Script de instalación del catálogo público BoConcept Crusardi
 * Configura automáticamente todos los componentes necesarios
 */

// Verificar que se ejecute desde el servidor
if (php_sapi_name() !== 'cli' && !isset($_GET['install'])) {
    die('Este script debe ejecutarse desde el servidor o con el parámetro install=true');
}

date_default_timezone_set('America/Bogota');

echo "=== INSTALADOR CATÁLOGO PÚBLICO BOCONCEPT CRUSARDI ===\n\n";

// Función para logging
function logInstall($message) {
    echo "[" . date('Y-m-d H:i:s') . "] $message\n";
}

// Función para verificar archivos
function checkFile($file) {
    if (file_exists($file)) {
        logInstall("✓ $file existe");
        return true;
    } else {
        logInstall("✗ $file NO existe");
        return false;
    }
}

// Función para crear directorio
function createDirectory($dir) {
    if (!is_dir($dir)) {
        if (mkdir($dir, 0755, true)) {
            logInstall("✓ Directorio creado: $dir");
            return true;
        } else {
            logInstall("✗ Error creando directorio: $dir");
            return false;
        }
    } else {
        logInstall("✓ Directorio ya existe: $dir");
        return true;
    }
}

// Función para verificar permisos
function checkPermissions($file) {
    if (is_writable($file)) {
        logInstall("✓ Permisos de escritura: $file");
        return true;
    } else {
        logInstall("✗ Sin permisos de escritura: $file");
        return false;
    }
}

// 1. Verificar archivos necesarios
logInstall("1. Verificando archivos necesarios...");
$required_files = [
    'index.php',
    'actualizar.php',
    'cron.php',
    'config.php'
];

$files_ok = true;
foreach ($required_files as $file) {
    if (!checkFile($file)) {
        $files_ok = false;
    }
}

if (!$files_ok) {
    logInstall("ERROR: Faltan archivos necesarios. Instalación abortada.");
    exit(1);
}

// 2. Crear directorios necesarios
logInstall("\n2. Creando directorios necesarios...");
$directories = [
    'logs',
    'cache',
    'backups'
];

foreach ($directories as $dir) {
    createDirectory($dir);
}

// 3. Verificar permisos
logInstall("\n3. Verificando permisos...");
$writable_dirs = [
    'logs',
    'cache',
    'backups'
];

$permissions_ok = true;
foreach ($writable_dirs as $dir) {
    if (!checkPermissions($dir)) {
        $permissions_ok = false;
    }
}

if (!$permissions_ok) {
    logInstall("ERROR: Problemas de permisos. Instalación abortada.");
    exit(1);
}

// 4. Verificar conexión a base de datos
logInstall("\n4. Verificando conexión a base de datos...");
try {
    require_once '/home/crusardi/config_crusardi/database.php';
    
    if ($pdo) {
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM stock_calculado_cache");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        logInstall("✓ Conexión a BD exitosa. Productos en cache: " . $result['total']);
    } else {
        logInstall("✗ Error de conexión a base de datos");
        exit(1);
    }
} catch (Exception $e) {
    logInstall("✗ Error de conexión a base de datos: " . $e->getMessage());
    exit(1);
}

// 5. Probar el catálogo público
logInstall("\n5. Probando catálogo público...");
$catalogo_url = 'https://stock.crusardi.net/catalogo_publico/';
$context = stream_context_create([
    'http' => [
        'timeout' => 30,
        'method' => 'GET'
    ]
]);

$response = @file_get_contents($catalogo_url, false, $context);
if ($response !== false) {
    logInstall("✓ Catálogo público accesible");
} else {
    logInstall("✗ Error accediendo al catálogo público");
}

// 6. Probar actualización automática
logInstall("\n6. Probando actualización automática...");
$update_url = 'https://stock.crusardi.net/catalogo_publico/actualizar.php';
$response = @file_get_contents($update_url, false, $context);

if ($response !== false) {
    $result = json_decode($response, true);
    if ($result && $result['success']) {
        logInstall("✓ Actualización automática funcionando");
        logInstall("  - Productos disponibles: " . $result['summary']['productos_disponibles']);
        logInstall("  - Total unidades: " . $result['summary']['total_unidades']);
    } else {
        logInstall("✗ Error en actualización automática: " . ($result['error'] ?? 'Error desconocido'));
    }
} else {
    logInstall("✗ No se pudo probar la actualización automática");
}

// 7. Configurar cron job
logInstall("\n7. Configurando cron job...");
$cron_url = 'https://stock.crusardi.net/catalogo_publico/cron.php?action=install';
$response = @file_get_contents($cron_url, false, $context);

if ($response !== false) {
    $result = json_decode($response, true);
    if ($result && $result['success']) {
        logInstall("✓ Cron job configurado exitosamente");
        logInstall("  - Entradas configuradas: " . $result['status']['total_entries']);
    } else {
        logInstall("✗ Error configurando cron job");
    }
} else {
    logInstall("✗ No se pudo configurar el cron job");
}

// 8. Crear archivo de configuración inicial
logInstall("\n8. Creando configuración inicial...");
$config_file = 'logs/catalogo_config.json';
$initial_config = [
    'installed' => true,
    'install_date' => date('Y-m-d H:i:s'),
    'version' => '1.0.0',
    'update_interval' => 6,
    'last_update' => date('Y-m-d H:i:s'),
    'status' => 'active'
];

file_put_contents($config_file, json_encode($initial_config, JSON_PRETTY_PRINT));
logInstall("✓ Configuración inicial creada");

// 9. Crear script de monitoreo
logInstall("\n9. Creando script de monitoreo...");
$monitor_script = '<?php
/**
 * Script de monitoreo del catálogo público
 */
require_once "config.php";

$config_file = __DIR__ . "/logs/catalogo_config.json";
$config = json_decode(file_get_contents($config_file), true);

echo "=== MONITOR CATÁLOGO PÚBLICO ===\n";
echo "Estado: " . ($config["status"] ?? "Desconocido") . "\n";
echo "Última actualización: " . ($config["last_update"] ?? "Nunca") . "\n";
echo "Versión: " . ($config["version"] ?? "Desconocida") . "\n";

// Verificar accesibilidad
$catalogo_url = "https://stock.crusardi.net/catalogo_publico/";
$response = @file_get_contents($catalogo_url, false, stream_context_create(["http" => ["timeout" => 10]]));

if ($response !== false) {
    echo "Catálogo: ✓ Accesible\n";
} else {
    echo "Catálogo: ✗ No accesible\n";
}

// Verificar logs recientes
$log_file = __DIR__ . "/logs/auto_update.log";
if (file_exists($log_file)) {
    $last_log = file_get_contents($log_file);
    $lines = explode("\n", $last_log);
    $last_line = end($lines);
    echo "Último log: " . $last_line . "\n";
} else {
    echo "Último log: No disponible\n";
}
?>';

file_put_contents('monitor.php', $monitor_script);
logInstall("✓ Script de monitoreo creado");

// 10. Resumen final
logInstall("\n=== INSTALACIÓN COMPLETADA ===");
logInstall("URL del catálogo: https://stock.crusardi.net/catalogo_publico/");
logInstall("Actualización automática: Cada 6 horas");
logInstall("Logs: logs/auto_update.log");
logInstall("Monitoreo: php monitor.php");
logInstall("Configuración: logs/catalogo_config.json");

logInstall("\n=== PRÓXIMOS PASOS ===");
logInstall("1. Verificar que el cron job esté funcionando: crontab -l");
logInstall("2. Monitorear los logs: tail -f logs/auto_update.log");
logInstall("3. Probar el catálogo: https://stock.crusardi.net/catalogo_publico/");
logInstall("4. Configurar notificaciones en config.php");

logInstall("\n¡Instalación completada exitosamente! 🎉");
?>
