<?php
// /stock/pages/usuarios.php

// Doble capa de seguridad: si un no-admin intenta acceder a esta página, lo detenemos.
if ($_SESSION['user_role'] !== 'admin') {
    die("<div class='p-4 bg-red-100 text-red-700 rounded'>Acceso denegado. Se requiere rol de administrador.</div>");
}

$error_message = null;
$success_message = null;

// --- LÓGICA PARA MANEJAR ACCIONES (CREAR, EDITAR, BORRAR) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    try {
        switch ($action) {
            case 'create':
                $stmt = $pdo->prepare("INSERT INTO users (email, name, role) VALUES (?, ?, ?)");
                $stmt->execute([trim($_POST['email']), trim($_POST['name']), $_POST['role']]);
                $success_message = "Usuario '" . htmlspecialchars($_POST['email']) . "' creado con éxito.";
                break;

            case 'update':
                $stmt = $pdo->prepare("UPDATE users SET name = ?, role = ? WHERE id = ?");
                $stmt->execute([trim($_POST['name']), $_POST['role'], $_POST['id']]);
                $success_message = "Usuario actualizado con éxito.";
                break;

            case 'delete':
                // Prevenir que un admin se borre a sí mismo
                if ($_POST['id'] == $_SESSION['user_id']) {
                    throw new Exception("No puedes eliminar tu propia cuenta de administrador.");
                }
                $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                $stmt->execute([$_POST['id']]);
                $success_message = "Usuario eliminado con éxito.";
                break;
        }
    } catch (Exception $e) {
        // Manejar errores, como intentar crear un email duplicado
        if ($e->getCode() == '23000') { // Código de error SQL para entrada duplicada
             $error_message = "Error: El correo electrónico '" . htmlspecialchars($_POST['email']) . "' ya existe.";
        } else {
             $error_message = "Ocurrió un error: " . $e->getMessage();
        }
    }
}

// --- OBTENER TODOS LOS USUARIOS PARA MOSTRARLOS EN LA TABLA ---
$users_stmt = $pdo->query("SELECT id, email, name, role, created_at FROM users ORDER BY name ASC");
$users = $users_stmt->fetchAll();

$roles_disponibles = ['admin', 'editor', 'viewer'];
?>

<div class="space-y-6">
    <header class="flex items-center justify-between gap-4">
        <div class="flex items-center gap-4">
            <i class="fas fa-users-cog fa-2x text-indigo-600"></i>
            <div>
                <h1 class="text-2xl font-bold">Administración de Usuarios</h1>
                <p class="text-gray-600">Añadir, editar o eliminar usuarios y sus roles.</p>
            </div>
        </div>
        <button onclick="openModal('create-modal')" class="bg-green-600 text-white font-semibold px-4 py-2 rounded-lg hover:bg-green-700">
            <i class="fas fa-plus mr-2"></i>Añadir Usuario
        </button>
    </header>

    <?php if ($success_message): ?>
        <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-r-lg" role="alert"><?php echo htmlspecialchars($success_message); ?></div>
    <?php endif; ?>
    <?php if ($error_message): ?>
         <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-r-lg" role="alert"><?php echo htmlspecialchars($error_message); ?></div>
    <?php endif; ?>

    <div class="bg-white p-6 rounded-lg shadow-md overflow-x-auto">
        <table class="w-full text-sm">
            <thead class="bg-gray-50">
                <tr>
                    <th class="p-3 text-left font-semibold text-gray-600">Nombre</th>
                    <th class="p-3 text-left font-semibold text-gray-600">Email</th>
                    <th class="p-3 text-left font-semibold text-gray-600">Rol</th>
                    <th class="p-3 text-right font-semibold text-gray-600">Acciones</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($users as $user): ?>
                <tr class="border-b hover:bg-gray-50">
                    <td class="p-3 font-medium"><?php echo htmlspecialchars($user['name']); ?></td>
                    <td class="p-3 text-gray-600"><?php echo htmlspecialchars($user['email']); ?></td>
                    <td class="p-3">
                        <span class="px-2 py-1 font-semibold leading-tight rounded-full
                            <?php echo $user['role'] === 'admin' ? 'bg-red-100 text-red-700' : ''; ?>
                            <?php echo $user['role'] === 'editor' ? 'bg-blue-100 text-blue-700' : ''; ?>
                            <?php echo $user['role'] === 'viewer' ? 'bg-gray-200 text-gray-700' : ''; ?>">
                            <?php echo ucfirst($user['role']); ?>
                        </span>
                    </td>
                    <td class="p-3 text-right">
                        <button onclick='openEditModal(<?php echo json_encode($user); ?>)' class="text-blue-600 hover:text-blue-800 mr-4" title="Editar"><i class="fas fa-pencil-alt"></i></button>
                        <?php if ($user['id'] != $_SESSION['user_id']): // No mostrar botón de borrar para el usuario actual ?>
                            <button onclick='openDeleteModal(<?php echo $user['id']; ?>, "<?php echo htmlspecialchars($user['email']); ?>")' class="text-red-600 hover:text-red-800" title="Eliminar"><i class="fas fa-trash"></i></button>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<div id="create-modal" class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center p-4 hidden">
    <div class="bg-white rounded-lg shadow-2xl w-full max-w-md p-6">
        <h3 class="text-xl font-bold mb-4">Añadir Nuevo Usuario</h3>
        <form action="index.php?page=usuarios" method="POST" class="space-y-4">
            <input type="hidden" name="action" value="create">
            <div>
                <label for="create-name" class="block text-sm font-medium text-gray-700">Nombre</label>
                <input type="text" id="create-name" name="name" class="mt-1 w-full p-2 border rounded-md" required>
            </div>
            <div>
                <label for="create-email" class="block text-sm font-medium text-gray-700">Email</label>
                <input type="email" id="create-email" name="email" class="mt-1 w-full p-2 border rounded-md" required>
            </div>
            <div>
                <label for="create-role" class="block text-sm font-medium text-gray-700">Rol</label>
                <select id="create-role" name="role" class="mt-1 w-full p-2 border rounded-md">
                    <?php foreach($roles_disponibles as $role): ?>
                    <option value="<?php echo $role; ?>"><?php echo ucfirst($role); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="flex justify-end gap-4 pt-4">
                <button type="button" onclick="closeModal('create-modal')" class="px-4 py-2 bg-gray-200 rounded-lg hover:bg-gray-300">Cancelar</button>
                <button type="submit" class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">Crear Usuario</button>
            </div>
        </form>
    </div>
</div>

<div id="edit-modal" class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center p-4 hidden">
    <div class="bg-white rounded-lg shadow-2xl w-full max-w-md p-6">
        <h3 class="text-xl font-bold mb-4">Editar Usuario</h3>
        <form action="index.php?page=usuarios" method="POST" class="space-y-4">
            <input type="hidden" name="action" value="update">
            <input type="hidden" id="edit-id" name="id">
            <div>
                <label class="block text-sm font-medium text-gray-700">Email (no se puede cambiar)</label>
                <input type="email" id="edit-email" class="mt-1 w-full p-2 border bg-gray-100 rounded-md" readonly>
            </div>
            <div>
                <label for="edit-name" class="block text-sm font-medium text-gray-700">Nombre</label>
                <input type="text" id="edit-name" name="name" class="mt-1 w-full p-2 border rounded-md" required>
            </div>
            <div>
                <label for="edit-role" class="block text-sm font-medium text-gray-700">Rol</label>
                <select id="edit-role" name="role" class="mt-1 w-full p-2 border rounded-md"></select>
            </div>
            <div class="flex justify-end gap-4 pt-4">
                <button type="button" onclick="closeModal('edit-modal')" class="px-4 py-2 bg-gray-200 rounded-lg hover:bg-gray-300">Cancelar</button>
                <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">Guardar Cambios</button>
            </div>
        </form>
    </div>
</div>

<div id="delete-modal" class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center p-4 hidden">
    <div class="bg-white rounded-lg shadow-2xl w-full max-w-md p-6">
        <h3 class="text-xl font-bold text-red-600 mb-2">Confirmar Eliminación</h3>
        <p class="text-gray-600 mb-6">¿Estás seguro de que deseas eliminar al usuario <strong id="delete-email-display"></strong>? Esta acción es irreversible.</p>
        <form action="index.php?page=usuarios" method="POST">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" id="delete-id" name="id">
            <div class="flex justify-end gap-4">
                <button type="button" onclick="closeModal('delete-modal')" class="px-4 py-2 bg-gray-200 rounded-lg hover:bg-gray-300">Cancelar</button>
                <button type="submit" class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">Sí, Eliminar</button>
            </div>
        </form>
    </div>
</div>

<script>
    const rolesDisponibles = <?php echo json_encode($roles_disponibles); ?>;

    function openModal(modalId) { document.getElementById(modalId).classList.remove('hidden'); }
    function closeModal(modalId) { document.getElementById(modalId).classList.add('hidden'); }

    function openEditModal(user) {
        document.getElementById('edit-id').value = user.id;
        document.getElementById('edit-email').value = user.email;
        document.getElementById('edit-name').value = user.name;

        const roleSelect = document.getElementById('edit-role');
        roleSelect.innerHTML = ''; // Limpiar opciones anteriores
        rolesDisponibles.forEach(role => {
            const option = document.createElement('option');
            option.value = role;
            option.textContent = role.charAt(0).toUpperCase() + role.slice(1);
            if (role === user.role) {
                option.selected = true;
            }
            // Prevenir que un admin se rebaje su propio rol si es el único admin
            if (user.id === <?php echo $_SESSION['user_id']; ?> && user.role === 'admin') {
                 if (role !== 'admin') option.disabled = true;
            }
            roleSelect.appendChild(option);
        });

        openModal('edit-modal');
    }

    function openDeleteModal(id, email) {
        document.getElementById('delete-id').value = id;
        document.getElementById('delete-email-display').textContent = email;
        openModal('delete-modal');
    }
</script>